import chokidar from 'chokidar';
import { exec } from 'child_process';
import { promisify } from 'util';

const execAsync = promisify(exec);

// Configuration
const DEBOUNCE_DELAY = 1000; // 1 second
const WATCH_PATHS = [
    'app/**/*.php',
    'config/**/*.php',
    'routes/**/*.php',
    'database/migrations/**/*.php',
    'database/seeders/**/*.php',
];

const IGNORE_PATTERNS = [
    '**/vendor/**',
    '**/node_modules/**',
    '**/storage/**',
    '**/bootstrap/cache/**',
];

// State
let reloadTimeout = null;
let isReloading = false;
let changedFiles = new Set();

// Colors for console output
const colors = {
    reset: '\x1b[0m',
    bright: '\x1b[1m',
    cyan: '\x1b[36m',
    green: '\x1b[32m',
    yellow: '\x1b[33m',
    red: '\x1b[31m',
    gray: '\x1b[90m',
};

function log(message, color = colors.reset) {
    const timestamp = new Date().toLocaleTimeString('id-ID');
    console.log(`${colors.gray}[${timestamp}]${colors.reset} ${color}${message}${colors.reset}`);
}

async function reloadOctane() {
    if (isReloading) {
        return;
    }

    isReloading = true;
    const files = Array.from(changedFiles);
    changedFiles.clear();

    log(`🔄 Reloading Octane workers... (${files.length} file(s) changed)`, colors.yellow);

    // Show changed files (limit to 5 for readability)
    const displayFiles = files.slice(0, 5);
    displayFiles.forEach(file => {
        log(`   • ${file}`, colors.gray);
    });
    if (files.length > 5) {
        log(`   ... and ${files.length - 5} more`, colors.gray);
    }

    try {
        const { stdout, stderr } = await execAsync('php artisan octane:reload');

        if (stderr) {
            log(`⚠️  Warning: ${stderr.trim()}`, colors.yellow);
        }

        log('✅ Octane workers reloaded successfully', colors.green);
    } catch (error) {
        log(`❌ Failed to reload Octane: ${error.message}`, colors.red);
    } finally {
        isReloading = false;
    }
}

function scheduleReload(filePath) {
    changedFiles.add(filePath);

    // Clear existing timeout
    if (reloadTimeout) {
        clearTimeout(reloadTimeout);
    }

    // Schedule new reload
    reloadTimeout = setTimeout(() => {
        reloadOctane();
    }, DEBOUNCE_DELAY);
}

// Initialize watcher
log('🚀 Starting Octane file watcher...', colors.cyan);
log(`📁 Watching: ${WATCH_PATHS.join(', ')}`, colors.gray);
log(`⏱️  Debounce delay: ${DEBOUNCE_DELAY}ms`, colors.gray);
log('', colors.reset);

const watcher = chokidar.watch(WATCH_PATHS, {
    ignored: IGNORE_PATTERNS,
    persistent: true,
    ignoreInitial: true,
    awaitWriteFinish: {
        stabilityThreshold: 100,
        pollInterval: 100,
    },
});

watcher
    .on('ready', () => {
        log('👀 Watching for PHP file changes...', colors.green);
    })
    .on('change', (path) => {
        log(`📝 File changed: ${path}`, colors.cyan);
        scheduleReload(path);
    })
    .on('add', (path) => {
        log(`➕ File added: ${path}`, colors.cyan);
        scheduleReload(path);
    })
    .on('unlink', (path) => {
        log(`➖ File removed: ${path}`, colors.cyan);
        scheduleReload(path);
    })
    .on('error', (error) => {
        log(`❌ Watcher error: ${error.message}`, colors.red);
    });

// Graceful shutdown
process.on('SIGINT', () => {
    log('\n👋 Shutting down watcher...', colors.yellow);
    watcher.close();
    process.exit(0);
});

process.on('SIGTERM', () => {
    log('\n👋 Shutting down watcher...', colors.yellow);
    watcher.close();
    process.exit(0);
});
