<?php

namespace Database\Seeders;

use App\Models\DetailPresensiNonAsrama;
use App\Models\JenisPresensiNonAsrama;
use App\Models\Kelas;
use App\Models\PresensiNonAsrama;
use App\Models\TahunAjaran;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Database\Seeder;

class PresensiNonAsramaSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        // Get active tahun ajaran
        $tahunAjaran = TahunAjaran::where('is_aktif', true)->first();
        if (! $tahunAjaran) {
            if ($this->command) {
                $this->command->error('No active TahunAjaran found. Cannot seed PresensiNonAsrama.');
            }

            return;
        }

        // Get all jenis presensi
        $jenisPresensiList = JenisPresensiNonAsrama::where('is_active', true)->get();
        if ($jenisPresensiList->isEmpty()) {
            if ($this->command) {
                $this->command->error('No JenisPresensiNonAsrama found. Cannot seed PresensiNonAsrama.');
            }

            return;
        }

        // Get all kelas
        $kelasList = Kelas::all();
        if ($kelasList->isEmpty()) {
            if ($this->command) {
                $this->command->error('No Kelas found. Cannot seed PresensiNonAsrama.');
            }

            return;
        }

        // Get a user for petugas
        $user = User::first();
        if (! $user) {
            if ($this->command) {
                $this->command->error('No User found. Cannot seed PresensiNonAsrama.');
            }

            return;
        }

        $totalPresensi = 0;
        $totalDetail = 0;

        // For each kelas, create presensi records
        foreach ($kelasList as $kelas) {
            if ($this->command) {
                $this->command->info("Creating presensi non-asrama for class: {$kelas->nama}");
            }

            // Generate 4 random dates (mix of 1 week and 1 month ago)
            $dates = $this->generateRandomDates(4);

            foreach ($dates as $tanggal) {
                // For each day, create entries for a few random types (e.g., Subuh & Dzuhur)
                // Pick 2 random types per day to simulate
                $dailyTypes = $jenisPresensiList->random(min(2, $jenisPresensiList->count()));

                foreach ($dailyTypes as $jenisPresensi) {
                    // Create PresensiNonAsrama
                    $presensi = PresensiNonAsrama::create([
                        'tanggal' => $tanggal,
                        'tahun_ajaran_id' => $tahunAjaran->id,
                        'jenis_presensi_non_asrama_id' => $jenisPresensi->id,
                        'kelas_id' => $kelas->id,
                        'user_id' => $user->id,
                        'keterangan' => fake()->optional(0.3)->sentence(),
                        'is_locked' => fake()->boolean(50), // 50% locked
                    ]);

                    $totalPresensi++;

                    // Get all siswa in this kelas
                    $siswaList = $kelas->siswas;

                    // Create DetailPresensiNonAsrama for each siswa
                    foreach ($siswaList as $siswa) {
                        // Random status with realistic distribution
                        $status = $this->getRandomStatus();

                        DetailPresensiNonAsrama::create([
                            'presensi_non_asrama_id' => $presensi->id,
                            'siswa_id' => $siswa->id,
                            'status' => $status,
                            'keterangan' => $status !== 'hadir' ? fake()->optional(0.5)->sentence() : null,
                        ]);

                        $totalDetail++;
                    }
                }
            }
        }

        if ($this->command) {
            $this->command->info("Successfully created {$totalPresensi} presensi non-asrama with {$totalDetail} detail records.");
        }
    }

    /**
     * Generate random dates
     */
    private function generateRandomDates(int $count): array
    {
        $dates = [];
        $now = Carbon::now();

        // Limit to 2 dates to reduce volume as requested (approx 10 total across 4 classes)
        // With 4 classes x 2 dates x 2 types = 16 presensi records (close to 10)
        $count = 2;

        for ($i = 0; $i < $count; $i++) {
            $daysAgo = fake()->numberBetween(1, 14); // Last 2 weeks
            $date = $now->copy()->subDays($daysAgo);

            // Skip weekends if strictly following school days, but prayers are everyday?
            // Assuming prayers are everyday for asrama, but for non-asrama maybe not?
            // Usually non-asrama only tracked on school days?
            // Let's assume everyday for now or school days. Usually school days.
            while ($date->dayOfWeek === 0) { // Skip Sunday
                $date->subDay();
            }

            $dates[] = $date->format('Y-m-d');
        }

        $dates = array_unique($dates);
        sort($dates);

        return array_slice($dates, 0, $count);
    }

    /**
     * Get random status with realistic distribution
     */
    private function getRandomStatus(): string
    {
        $rand = fake()->numberBetween(1, 100);

        if ($rand <= 85) {
            return 'hadir';
        } elseif ($rand <= 93) {
            return 'sakit';
        } elseif ($rand <= 98) {
            return 'izin';
        } else {
            return 'alpha';
        }
    }
}
