<?php

namespace Database\Seeders;

use App\Models\DetailPresensiAsrama;
use App\Models\JenisPresensiAsrama;
use App\Models\Kamar;
use App\Models\PresensiAsrama;
use App\Models\TahunAjaran;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Database\Seeder;

class PresensiAsramaSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        // Get active tahun ajaran
        $tahunAjaran = TahunAjaran::where('is_aktif', true)->first();
        if (! $tahunAjaran) {
            if ($this->command) {
                $this->command->error('No active TahunAjaran found. Cannot seed PresensiAsrama.');
            }

            return;
        }

        // Get all jenis presensi
        $jenisPresensiList = JenisPresensiAsrama::where('is_active', true)->get();
        if ($jenisPresensiList->isEmpty()) {
            if ($this->command) {
                $this->command->error('No JenisPresensiAsrama found. Cannot seed PresensiAsrama.');
            }

            return;
        }

        // Get all kamar
        $kamarList = Kamar::all();
        if ($kamarList->isEmpty()) {
            if ($this->command) {
                $this->command->error('No Kamar found. Cannot seed PresensiAsrama.');
            }

            return;
        }

        // Get a user for petugas
        $user = User::first();
        if (! $user) {
            if ($this->command) {
                $this->command->error('No User found. Cannot seed PresensiAsrama.');
            }

            return;
        }

        $totalPresensi = 0;
        $totalDetail = 0;

        // For each kamar, create presensi records
        foreach ($kamarList as $kamar) {
            if ($this->command) {
                $this->command->info("Creating presensi for kamar: {$kamar->nama_kamar}");
            }

            // Generate 3 random dates (1 week ago)
            $dates = $this->generateRandomDates(3);

            foreach ($dates as $tanggal) {
                // For each date, create 3 types of presensi (Subuh, Maghrib, Isya)
                // We pick these 3 to simulate daily routine
                $dailyTypes = $jenisPresensiList->whereIn('nama', ['Shalat Subuh', 'Shalat Maghrib', 'Shalat Isya']);

                foreach ($dailyTypes as $jenisPresensi) {
                    // Create PresensiAsrama
                    $presensi = PresensiAsrama::create([
                        'tanggal' => $tanggal,
                        'tahun_ajaran_id' => $tahunAjaran->id,
                        'jenis_presensi_asrama_id' => $jenisPresensi->id,
                        'kamar_id' => $kamar->id,
                        'user_id' => $user->id,
                        'keterangan' => fake()->optional(0.2)->sentence(),
                        'is_locked' => fake()->boolean(70), // 70% locked
                    ]);

                    $totalPresensi++;

                    // Get all siswa in this kamar
                    $siswaList = $kamar->siswas;

                    // Create DetailPresensiAsrama for each siswa
                    foreach ($siswaList as $siswa) {
                        // Random status
                        $status = $this->getRandomStatus();

                        DetailPresensiAsrama::create([
                            'presensi_asrama_id' => $presensi->id,
                            'siswa_id' => $siswa->id,
                            'status' => $status,
                            'keterangan' => $status !== 'hadir' ? fake()->optional(0.5)->sentence() : null,
                        ]);

                        $totalDetail++;
                    }
                }
            }
        }

        if ($this->command) {
            $this->command->info("Successfully created {$totalPresensi} presensi asrama with {$totalDetail} detail records.");
        }
    }

    /**
     * Generate random dates (1 week ago)
     */
    private function generateRandomDates(int $count): array
    {
        $dates = [];
        $now = Carbon::now();

        for ($i = 0; $i < $count; $i++) {
            $daysAgo = fake()->numberBetween(1, 7);
            $date = $now->copy()->subDays($daysAgo);
            $dates[] = $date->format('Y-m-d');
        }

        // Remove duplicates and sort
        $dates = array_unique($dates);
        sort($dates);

        return array_slice($dates, 0, $count);
    }

    /**
     * Get random status with realistic distribution
     * 90% hadir, 5% sakit, 3% izin, 2% alpha
     */
    private function getRandomStatus(): string
    {
        $rand = fake()->numberBetween(1, 100);

        if ($rand <= 90) {
            return 'hadir';
        } elseif ($rand <= 95) {
            return 'sakit';
        } elseif ($rand <= 98) {
            return 'izin';
        } else {
            return 'alpha';
        }
    }
}
