<?php

namespace App\Providers\Filament;

use App\Filament\Pages\Login;
use App\Filament\Widgets\PresensiNonAsramaWidget;
use App\Models\User;
use App\Settings\KaidoSetting;
use BezhanSalleh\FilamentShield\FilamentShieldPlugin;
use DutchCodingCompany\FilamentSocialite\FilamentSocialitePlugin;
use DutchCodingCompany\FilamentSocialite\Provider;
use Filament\Forms\Components\FileUpload;
use Filament\Http\Middleware\Authenticate;
use Filament\Http\Middleware\DisableBladeIconComponents;
use Filament\Http\Middleware\DispatchServingFilamentEvent;
use Filament\Pages;
use Filament\Panel;
use Filament\PanelProvider;
use Filament\Support\Colors\Color;
use Filament\Widgets;
use Illuminate\Cookie\Middleware\AddQueuedCookiesToResponse;
use Illuminate\Cookie\Middleware\EncryptCookies;
use Illuminate\Foundation\Http\Middleware\VerifyCsrfToken;
use Illuminate\Routing\Middleware\SubstituteBindings;
use Illuminate\Session\Middleware\AuthenticateSession;
use Illuminate\Session\Middleware\StartSession;
use Illuminate\View\Middleware\ShareErrorsFromSession;
use Jeffgreco13\FilamentBreezy\BreezyCore;
use Laravel\Socialite\Contracts\User as SocialiteUserContract;
use Rupadana\ApiService\ApiServicePlugin;

class AdminPanelProvider extends PanelProvider
{
    // Cache untuk asset paths (tidak berubah antar request)
    private static ?array $assetCache = null;

    /**
     * Get settings instance fresh each time (Octane-safe)
     * Tidak menyimpan di property untuk menghindari memory leak
     */
    private function getSettings(): ?KaidoSetting
    {
        static $tableExists = null;

        // Cache table check only (bukan instance-nya)
        if ($tableExists === null) {
            try {
                $tableExists = \Illuminate\Support\Facades\Schema::hasTable('settings');
            } catch (\Exception $e) {
                $tableExists = false;
            }
        }

        if (! $tableExists) {
            return null;
        }

        // Selalu ambil fresh instance
        try {
            return app(KaidoSetting::class);
        } catch (\Exception $e) {
            return null;
        }
    }

    public function panel(Panel $panel): Panel
    {
        $settings = $this->getSettings();

        return $panel
            ->default()
            ->id('admin')
            ->brandName('AbsensiKauman')

            ->favicon($this->getFavicon())

            ->path('')
            ->breadcrumbs(false)
            ->when($settings->login_enabled ?? true, fn ($panel) => $panel->login(Login::class))
            ->when($settings->registration_enabled ?? true, fn ($panel) => $panel->registration())
            ->when($settings->password_reset_enabled ?? true, fn ($panel) => $panel->passwordReset())
            ->colors([
                // 'primary' => Color::Blue,
                'danger' => Color::Rose,
                'gray' => Color::Gray,
                'info' => Color::Blue,
                'primary' => Color::Blue,
                'success' => Color::Emerald,
                'warning' => Color::Orange,
            ])

            ->discoverResources(in: app_path('Filament/Resources'), for: 'App\\Filament\\Resources')
            ->discoverPages(in: app_path('Filament/Pages'), for: 'App\\Filament\\Pages')
            ->pages([
                Pages\Dashboard::class,
            ])
            ->discoverWidgets(in: app_path('Filament/Widgets'), for: 'App\\Filament\\Widgets')
            ->widgets([
                Widgets\AccountWidget::class,
                PresensiNonAsramaWidget::class,
            ])
            ->navigationGroups([
                'Dashboard',
                'Akademik',
                'Presensi Madrasah',
                'Kegiatan Pengasuhan',
                'Laporan',
                'Settings',
            ])
            ->middleware([
                EncryptCookies::class,
                AddQueuedCookiesToResponse::class,
                StartSession::class,
                AuthenticateSession::class,
                ShareErrorsFromSession::class,
                VerifyCsrfToken::class,
                SubstituteBindings::class,
                DisableBladeIconComponents::class,
                DispatchServingFilamentEvent::class,
            ])
            ->sidebarCollapsibleOnDesktop(true)
            ->authMiddleware([
                Authenticate::class,
            ])
            ->plugins(
                $this->getPlugins($settings)
            )
            ->databaseNotifications();
    }

    private function getPlugins(?KaidoSetting $settings): array
    {
        $plugins = [
            FilamentShieldPlugin::make()
                ->gridColumns([
                    'default' => 1,
                    'sm' => 2,
                    'lg' => 3,
                ])
                ->sectionColumnSpan(1)
                ->checkboxListColumns([
                    'default' => 1,
                    'sm' => 2,
                    'lg' => 4,
                ])
                ->resourceCheckboxListColumns([
                    'default' => 1,
                    'sm' => 2,
                ]),
            ApiServicePlugin::make(),
            BreezyCore::make()
                ->myProfile(
                    shouldRegisterUserMenu: true,
                    shouldRegisterNavigation: true,
                    navigationGroup: 'Settings',
                    hasAvatars: true,
                    slug: 'my-profile'
                )
                ->avatarUploadComponent(fn ($fileUpload) => $fileUpload->disableLabel())
                ->avatarUploadComponent(
                    fn () => FileUpload::make('avatar_url')
                        ->image()
                        ->disk('public')
                )
                ->enableTwoFactorAuthentication(),
        ];

        if ($settings->sso_enabled ?? true) {
            $plugins[] =
                FilamentSocialitePlugin::make()
                    ->providers([
                        Provider::make('google')
                            ->label('Google')
                            ->icon('fab-google')
                            ->color(Color::hex('#2f2a6b'))
                            ->outlined(true)
                            ->stateless(false),
                    ])->registration(true)
                    ->createUserUsing(function (string $provider, SocialiteUserContract $oauthUser, FilamentSocialitePlugin $plugin) {
                        $user = User::firstOrNew([
                            'email' => $oauthUser->getEmail(),
                        ]);
                        $user->name = $oauthUser->getName();
                        $user->email = $oauthUser->getEmail();
                        $user->email_verified_at = now();
                        $user->save();

                        return $user;
                    });
        }

        return $plugins;
    }

    /**
     * Initialize asset cache once (Octane-safe - static cache)
     */
    private function initAssetCache(): void
    {
        if (self::$assetCache !== null) {
            return;
        }

        self::$assetCache = [
            'brandLogo' => null,
            'darkLogo' => null,
            'favicon' => null,
        ];

        // Brand logo
        if (file_exists(public_path('images/logo-light.png'))) {
            self::$assetCache['brandLogo'] = asset('images/logo-light.png');
        } elseif (file_exists(public_path('asset/logo.png'))) {
            self::$assetCache['brandLogo'] = asset('asset/logo.png');
        }

        // Dark logo
        if (file_exists(public_path('images/logo-dark.png'))) {
            self::$assetCache['darkLogo'] = asset('images/logo-dark.png');
        } else {
            self::$assetCache['darkLogo'] = self::$assetCache['brandLogo'];
        }

        // Favicon
        if (file_exists(public_path('images/favicon.png'))) {
            self::$assetCache['favicon'] = asset('images/favicon.png');
        } elseif (file_exists(public_path('favicon.ico'))) {
            self::$assetCache['favicon'] = asset('favicon.ico');
        }
    }

    private function getBrandLogo(): ?string
    {
        $this->initAssetCache();

        return self::$assetCache['brandLogo'];
    }

    private function getDarkModeBrandLogo(): ?string
    {
        $this->initAssetCache();

        return self::$assetCache['darkLogo'];
    }

    private function getFavicon(): ?string
    {
        $this->initAssetCache();

        return self::$assetCache['favicon'];
    }
}
