<?php

namespace App\Imports;

use App\Models\Kamar;
use App\Models\Kelas;
use App\Models\Siswa;
use Carbon\Carbon;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Auth;
use Maatwebsite\Excel\Concerns\SkipsEmptyRows;
use Maatwebsite\Excel\Concerns\ToCollection;
use Maatwebsite\Excel\Concerns\WithBatchInserts;
use Maatwebsite\Excel\Concerns\WithChunkReading;
use Maatwebsite\Excel\Concerns\WithHeadingRow;

class SiswaImport implements SkipsEmptyRows, ToCollection, WithBatchInserts, WithChunkReading, WithHeadingRow
{
    protected $madrasahId;

    protected $errors = [];

    protected $successCount = 0;

    protected $failedCount = 0;

    protected $processedNisn = [];  // Track NISN yang sudah diproses dalam import ini

    public function __construct($madrasahId = null)
    {
        $this->madrasahId = $madrasahId ?? Auth::user()->madrasah_id;
    }

    public function collection(Collection $rows)
    {
        foreach ($rows as $index => $row) {
            try {
                // Debug: Log header keys di baris pertama
                if ($index === 0) {
                    \Log::info('Excel headers:', array_keys($row->toArray()));
                }

                // Helper function untuk convert scientific notation ke string
                $convertToString = function ($value) {
                    if (empty($value)) {
                        return null;
                    }
                    
                    // Convert ke string dulu
                    $value = (string) $value;
                    
                    // Hapus prefix apostrof jika ada (dari Excel)
                    $value = ltrim($value, "'");
                    
                    // Jika numeric dan panjang, convert ke string tanpa scientific notation
                    if (is_numeric($value)) {
                        return sprintf('%.0f', $value);
                    }

                    return $value;
                };

                // Helper untuk get value dengan fallback ke berbagai format header
                $getValue = function ($row, $keys) {
                    foreach ((array) $keys as $key) {
                        if (isset($row[$key]) && ! empty($row[$key])) {
                            return $row[$key];
                        }
                    }

                    return null;
                };

                // Ambil madrasah_id dari Excel (prioritas utama) atau gunakan default
                $madrasahId = $getValue($row, ['madrasah_id']) ?? $this->madrasahId;

                // Convert NISN, NIK, dan nomor lainnya ke string
                // Support multiple header formats
                $nisn = $convertToString($getValue($row, ['nisn', 'nisn_']));
                $nis = $convertToString($getValue($row, ['nis']));
                $nik = $convertToString($getValue($row, ['nik']));
                $nama = $getValue($row, ['nama', 'nama_']);
                $jenisKelamin = $getValue($row, ['jenis_kelamin', 'jenis_kelamin_lp']);
                $tempatLahir = $getValue($row, ['tempat_lahir']);
                $tanggalLahir = $getValue($row, ['tanggal_lahir', 'tanggal_lahir_yyyy_mm_dd']);
                $namaKelas = $getValue($row, ['nama_kelas', 'nama_kelas_']);
                $statusTempatTinggal = $getValue($row, ['status_tempat_tinggal', 'status_tempat_tinggal_asramanon_asrama']);
                $namaKamar = $getValue($row, ['nama_kamar']);
                $noKk = $convertToString($getValue($row, ['no_kk']));
                $namaAyah = $getValue($row, ['nama_ayah']);
                $nikAyah = $convertToString($getValue($row, ['nik_ayah']));
                $pendidikanAyah = $getValue($row, ['pendidikan_ayah']);
                $pekerjaanAyah = $getValue($row, ['pekerjaan_ayah']);
                $namaIbu = $getValue($row, ['nama_ibu']);
                $nikIbu = $convertToString($getValue($row, ['nik_ibu']));
                $pendidikanIbu = $getValue($row, ['pendidikan_ibu']);
                $pekerjaanIbu = $getValue($row, ['pekerjaan_ibu']);
                $noHpOrtu = $convertToString($getValue($row, ['no_hp_ortu', 'no_hp_orang_tua']));
                $alamat = $getValue($row, ['alamat']);
                $provinsi = $getValue($row, ['provinsi']);
                $kabupatenKota = $getValue($row, ['kabupaten_kota', 'kabupatenkota']);
                $kecamatan = $getValue($row, ['kecamatan']);
                $desaKelurahan = $getValue($row, ['desa_kelurahan', 'desakelurahan']);
                $kodePos = $getValue($row, ['kode_pos']);

                // Validasi manual
                if (empty($nisn)) {
                    $this->errors[] = 'Baris '.($index + 2).': NISN wajib diisi';
                    $this->failedCount++;

                    continue;
                }

                // Cek duplikat NISN dalam import session ini
                $nisnKey = $nisn.'_'.$madrasahId;
                if (isset($this->processedNisn[$nisnKey])) {
                    $this->errors[] = 'Baris '.($index + 2).": NISN '{$nisn}' sudah ada di baris {$this->processedNisn[$nisnKey]} (duplikat dalam file Excel)";
                    $this->failedCount++;

                    continue;
                }

                if (empty($nama)) {
                    $this->errors[] = 'Baris '.($index + 2).': Nama wajib diisi';
                    $this->failedCount++;

                    continue;
                }

                if (empty($jenisKelamin)) {
                    $this->errors[] = 'Baris '.($index + 2).': Jenis kelamin wajib diisi';
                    $this->failedCount++;

                    continue;
                }

                if (! in_array(strtoupper($jenisKelamin), ['L', 'P'])) {
                    $this->errors[] = 'Baris '.($index + 2).': Jenis kelamin harus L atau P';
                    $this->failedCount++;

                    continue;
                }

                if (empty($namaKelas)) {
                    $this->errors[] = 'Baris '.($index + 2).': Nama kelas wajib diisi';
                    $this->failedCount++;

                    continue;
                }

                // Cari kelas berdasarkan nama dan madrasah_id
                $kelas = Kelas::where('nama', $namaKelas)
                    ->where('madrasah_id', $madrasahId)
                    ->first();

                if (! $kelas) {
                    $this->errors[] = 'Baris '.($index + 2).": Kelas '{$namaKelas}' tidak ditemukan untuk madrasah_id {$madrasahId}";
                    $this->failedCount++;

                    continue;
                }

                // Cari kamar jika ada (opsional)
                $kamarId = null;
                if (! empty($namaKamar)) {
                    $kamar = Kamar::where('nama_kamar', $namaKamar)->first();
                    if ($kamar) {
                        $kamarId = $kamar->id;
                    }
                }

                // Parse tanggal lahir
                $tanggalLahirParsed = null;
                if (! empty($tanggalLahir)) {
                    try {
                        $tanggalLahirParsed = Carbon::parse($tanggalLahir)->format('Y-m-d');
                    } catch (\Exception $e) {
                        $this->errors[] = 'Baris '.($index + 2).': Format tanggal lahir tidak valid';
                    }
                }

                // Create atau update siswa
                Siswa::updateOrCreate(
                    [
                        'nisn' => $nisn,
                        'madrasah_id' => $madrasahId,
                    ],
                    [
                        'kelas_id' => $kelas->id,
                        'nis' => $nis,
                        'nik' => $nik,
                        'nama' => $nama,
                        'jenis_kelamin' => strtoupper($jenisKelamin),
                        'tempat_lahir' => $tempatLahir,
                        'tanggal_lahir' => $tanggalLahirParsed,
                        'status_tempat_tinggal' => $statusTempatTinggal ?? 'non_asrama',
                        'kamar_id' => $kamarId,
                        'no_kk' => $noKk,
                        'nama_ayah' => $namaAyah,
                        'nik_ayah' => $nikAyah,
                        'pendidikan_ayah' => $pendidikanAyah,
                        'pekerjaan_ayah' => $pekerjaanAyah,
                        'nama_ibu' => $namaIbu,
                        'nik_ibu' => $nikIbu,
                        'pendidikan_ibu' => $pendidikanIbu,
                        'pekerjaan_ibu' => $pekerjaanIbu,
                        'no_hp_ortu' => $noHpOrtu,
                        'alamat' => $alamat,
                        'provinsi' => $provinsi,
                        'kabupaten_kota' => $kabupatenKota,
                        'kecamatan' => $kecamatan,
                        'desa_kelurahan' => $desaKelurahan,
                        'kode_pos' => $kodePos,
                    ]
                );

                // Track NISN yang sudah diproses
                $this->processedNisn[$nisnKey] = $index + 2;

                $this->successCount++;
            } catch (\Exception $e) {
                $this->errors[] = 'Baris '.($index + 2).': '.$e->getMessage();
                $this->failedCount++;
            }
        }
    }

    public function batchSize(): int
    {
        return 100;
    }

    public function chunkSize(): int
    {
        return 100;
    }

    public function getErrors()
    {
        return $this->errors;
    }

    public function getSuccessCount()
    {
        return $this->successCount;
    }

    public function getFailedCount()
    {
        return $this->failedCount;
    }
}
