<?php

namespace App\Filament\Widgets;

use App\Models\Siswa;
use BezhanSalleh\FilamentShield\Traits\HasWidgetShield;
use Carbon\Carbon;
use Filament\Tables;
use Filament\Tables\Table;
use Filament\Widgets\TableWidget as BaseWidget;
use Illuminate\Support\Facades\Auth;

class SiswaBermasalahWidget extends BaseWidget
{
    use \App\Traits\HasStrictWidgetShield;

    protected static ?int $sort = 3;

    protected int|string|array $columnSpan = 'full';

    public function table(Table $table): Table
    {
        $startDate = Carbon::now()->subDays(30);
        $endDate = Carbon::now();

        return $table
            ->query(
                Siswa::query()
                    ->with(['madrasah', 'kelas'])
                    ->withCount([
                        'detailPresensiMadrasahs as total_presensi_madrasah' => function ($query) use ($startDate, $endDate) {
                            $query->whereHas('presensiMadrasah', function ($q) use ($startDate, $endDate) {
                                $q->whereBetween('tanggal', [$startDate, $endDate]);
                            });
                        },
                        'detailPresensiMadrasahs as hadir_madrasah' => function ($query) use ($startDate, $endDate) {
                            $query->where('status', 'hadir')
                                ->whereHas('presensiMadrasah', function ($q) use ($startDate, $endDate) {
                                    $q->whereBetween('tanggal', [$startDate, $endDate]);
                                });
                        },
                        'detailPresensiMadrasahs as alpha_madrasah' => function ($query) use ($startDate, $endDate) {
                            $query->where('status', 'alpha')
                                ->whereHas('presensiMadrasah', function ($q) use ($startDate, $endDate) {
                                    $q->whereBetween('tanggal', [$startDate, $endDate]);
                                });
                        },
                        'detailPresensiAsramas as total_presensi_asrama' => function ($query) use ($startDate, $endDate) {
                            $query->whereHas('presensiAsrama', function ($q) use ($startDate, $endDate) {
                                $q->whereBetween('tanggal', [$startDate, $endDate]);
                            });
                        },
                        'detailPresensiAsramas as hadir_asrama' => function ($query) use ($startDate, $endDate) {
                            $query->where('status', 'hadir')
                                ->whereHas('presensiAsrama', function ($q) use ($startDate, $endDate) {
                                    $q->whereBetween('tanggal', [$startDate, $endDate]);
                                });
                        },
                        'detailPresensiAsramas as alpha_asrama' => function ($query) use ($startDate, $endDate) {
                            $query->where('status', 'alpha')
                                ->whereHas('presensiAsrama', function ($q) use ($startDate, $endDate) {
                                    $q->whereBetween('tanggal', [$startDate, $endDate]);
                                });
                        },
                    ])
                    ->having(function ($query) {
                        // Filter siswa dengan tingkat kehadiran < 75%
                        $query->havingRaw('
                            CASE 
                                WHEN (total_presensi_madrasah + total_presensi_asrama) > 0 
                                THEN ((hadir_madrasah + hadir_asrama) / (total_presensi_madrasah + total_presensi_asrama) * 100) < 75
                                ELSE false
                            END
                        ');
                    })
                    ->orderByRaw('((hadir_madrasah + hadir_asrama) / NULLIF((total_presensi_madrasah + total_presensi_asrama), 0) * 100) ASC')
                    ->limit(10)
            )
            ->heading('Siswa dengan Kehadiran Rendah (< 75%)')
            ->description('10 siswa dengan tingkat kehadiran terendah dalam 30 hari terakhir')
            ->columns([
                Tables\Columns\TextColumn::make('nisn')
                    ->label('NISN')
                    ->searchable()
                    ->sortable(),

                Tables\Columns\TextColumn::make('nama')
                    ->label('Nama')
                    ->searchable()
                    ->sortable()
                    ->weight('bold'),

                Tables\Columns\TextColumn::make('madrasah.nama')
                    ->label('Madrasah')
                    ->sortable(),

                Tables\Columns\TextColumn::make('kelas.nama')
                    ->label('Kelas')
                    ->sortable(),

                Tables\Columns\TextColumn::make('tingkat_kehadiran')
                    ->label('Tingkat Kehadiran')
                    ->state(function (Siswa $record): string {
                        $totalPresensi = $record->total_presensi_madrasah + $record->total_presensi_asrama;
                        $totalHadir = $record->hadir_madrasah + $record->hadir_asrama;

                        if ($totalPresensi == 0) {
                            return '0%';
                        }

                        $persentase = round(($totalHadir / $totalPresensi) * 100, 1);

                        return $persentase.'%';
                    })
                    ->badge()
                    ->color(fn (Siswa $record): string => (($record->hadir_madrasah + $record->hadir_asrama) / max(($record->total_presensi_madrasah + $record->total_presensi_asrama), 1) * 100) >= 60
                            ? 'warning'
                            : 'danger'
                    )
                    ->sortable(),

                Tables\Columns\TextColumn::make('total_alpha')
                    ->label('Total Alpha')
                    ->state(function (Siswa $record): string {
                        return ($record->alpha_madrasah + $record->alpha_asrama).'x';
                    })
                    ->badge()
                    ->color('danger'),
            ])
            ->actions([
                Tables\Actions\Action::make('view')
                    ->label('Lihat Detail')
                    ->icon('heroicon-o-eye')
                    ->url(fn (Siswa $record): string => route('filament.admin.resources.siswas.edit', $record))
                    ->openUrlInNewTab(),
            ]);
    }
}
