<?php

namespace App\Filament\Widgets;

use App\Models\DetailPresensiAsrama;
use App\Models\DetailPresensiMadrasah;
use App\Models\Kamar;
use App\Models\Kelas;
use App\Models\Madrasah;
use App\Models\Siswa;
use BezhanSalleh\FilamentShield\Traits\HasWidgetShield;
use Carbon\Carbon;
use Filament\Widgets\StatsOverviewWidget as BaseWidget;
use Filament\Widgets\StatsOverviewWidget\Stat;
use Illuminate\Support\Facades\Auth;

class QuickStatsWidget extends BaseWidget
{
    use \App\Traits\HasStrictWidgetShield;

    protected static ?int $sort = 1;

    protected function getStats(): array
    {
        $user = Auth::user();
        $stats = [];

        // Stats untuk Walikelas
        if ($user->hasRole('walikelas')) {
            $kelas = Kelas::where('wali_kelas_id', $user->id)->first();

            if ($kelas) {
                $totalSiswa = Siswa::where('kelas_id', $kelas->id)->count();

                // Kehadiran hari ini
                $today = Carbon::today();
                $presensiToday = DetailPresensiMadrasah::whereHas('presensiMadrasah', function ($query) use ($kelas, $today) {
                    $query->where('kelas_id', $kelas->id)
                        ->where('tanggal', $today);
                })->count();

                $hadirToday = DetailPresensiMadrasah::whereHas('presensiMadrasah', function ($query) use ($kelas, $today) {
                    $query->where('kelas_id', $kelas->id)
                        ->where('tanggal', $today);
                })->where('status', 'hadir')->count();

                $persentaseHadirToday = $presensiToday > 0 ? round(($hadirToday / $presensiToday) * 100, 1) : 0;

                // Kehadiran bulan ini
                $startOfMonth = Carbon::now()->startOfMonth();
                $endOfMonth = Carbon::now()->endOfMonth();

                $presensiBulanIni = DetailPresensiMadrasah::whereHas('presensiMadrasah', function ($query) use ($kelas, $startOfMonth, $endOfMonth) {
                    $query->where('kelas_id', $kelas->id)
                        ->whereBetween('tanggal', [$startOfMonth, $endOfMonth]);
                })->count();

                $hadirBulanIni = DetailPresensiMadrasah::whereHas('presensiMadrasah', function ($query) use ($kelas, $startOfMonth, $endOfMonth) {
                    $query->where('kelas_id', $kelas->id)
                        ->whereBetween('tanggal', [$startOfMonth, $endOfMonth]);
                })->where('status', 'hadir')->count();

                $persentaseHadirBulanIni = $presensiBulanIni > 0 ? round(($hadirBulanIni / $presensiBulanIni) * 100, 1) : 0;

                $stats[] = Stat::make('Kelas Anda', $kelas->nama)
                    ->description('Total '.$totalSiswa.' siswa')
                    ->descriptionIcon('heroicon-o-academic-cap')
                    ->color('primary');

                $stats[] = Stat::make('Kehadiran Hari Ini', $persentaseHadirToday.'%')
                    ->description($hadirToday.' dari '.$presensiToday.' siswa hadir')
                    ->descriptionIcon('heroicon-o-check-circle')
                    ->color($persentaseHadirToday >= 80 ? 'success' : ($persentaseHadirToday >= 60 ? 'warning' : 'danger'));

                $stats[] = Stat::make('Kehadiran Bulan Ini', $persentaseHadirBulanIni.'%')
                    ->description('Rata-rata kehadiran')
                    ->descriptionIcon('heroicon-o-chart-bar')
                    ->color($persentaseHadirBulanIni >= 80 ? 'success' : ($persentaseHadirBulanIni >= 60 ? 'warning' : 'danger'))
                    ->chart([70, 75, 78, 80, 82, $persentaseHadirBulanIni]);
            } else {
                $stats[] = Stat::make('Info', 'Anda belum ditugaskan di kelas manapun')
                    ->color('gray');
            }
        }

        // Stats untuk Wali Kamar
        if ($user->hasRole('wali kamar')) {
            $kamar = Kamar::where('wali_kamar_id', $user->id)->first();

            if ($kamar) {
                $totalSiswa = Siswa::where('kamar_id', $kamar->id)->count();

                // Kehadiran hari ini
                $today = Carbon::today();
                $presensiToday = DetailPresensiAsrama::whereHas('presensiAsrama', function ($query) use ($kamar, $today) {
                    $query->where('kamar_id', $kamar->id)
                        ->where('tanggal', $today);
                })->count();

                $hadirToday = DetailPresensiAsrama::whereHas('presensiAsrama', function ($query) use ($kamar, $today) {
                    $query->where('kamar_id', $kamar->id)
                        ->where('tanggal', $today);
                })->where('status', 'hadir')->count();

                $persentaseHadirToday = $presensiToday > 0 ? round(($hadirToday / $presensiToday) * 100, 1) : 0;

                // Kehadiran bulan ini
                $startOfMonth = Carbon::now()->startOfMonth();
                $endOfMonth = Carbon::now()->endOfMonth();

                $presensiBulanIni = DetailPresensiAsrama::whereHas('presensiAsrama', function ($query) use ($kamar, $startOfMonth, $endOfMonth) {
                    $query->where('kamar_id', $kamar->id)
                        ->whereBetween('tanggal', [$startOfMonth, $endOfMonth]);
                })->count();

                $hadirBulanIni = DetailPresensiAsrama::whereHas('presensiAsrama', function ($query) use ($kamar, $startOfMonth, $endOfMonth) {
                    $query->where('kamar_id', $kamar->id)
                        ->whereBetween('tanggal', [$startOfMonth, $endOfMonth]);
                })->where('status', 'hadir')->count();

                $persentaseHadirBulanIni = $presensiBulanIni > 0 ? round(($hadirBulanIni / $presensiBulanIni) * 100, 1) : 0;

                $stats[] = Stat::make('Kamar Anda', $kamar->nama_kamar)
                    ->description('Total '.$totalSiswa.' siswa')
                    ->descriptionIcon('heroicon-o-home')
                    ->color('primary');

                $stats[] = Stat::make('Kehadiran Hari Ini', $persentaseHadirToday.'%')
                    ->description($hadirToday.' dari '.$presensiToday.' siswa hadir')
                    ->descriptionIcon('heroicon-o-check-circle')
                    ->color($persentaseHadirToday >= 80 ? 'success' : ($persentaseHadirToday >= 60 ? 'warning' : 'danger'));

                $stats[] = Stat::make('Kehadiran Bulan Ini', $persentaseHadirBulanIni.'%')
                    ->description('Rata-rata kehadiran')
                    ->descriptionIcon('heroicon-o-chart-bar')
                    ->color($persentaseHadirBulanIni >= 80 ? 'success' : ($persentaseHadirBulanIni >= 60 ? 'warning' : 'danger'))
                    ->chart([70, 75, 78, 80, 82, $persentaseHadirBulanIni]);
            } else {
                $stats[] = Stat::make('Info', 'Anda belum ditugaskan di kamar manapun')
                    ->color('gray');
            }
        }

        // Stats untuk Kepala Sekolah
        if ($user->hasRole('kepala sekolah')) {
            // Cari madrasah yang dipimpin oleh kepala sekolah ini
            $madrasah = Madrasah::where('kepala_sekolah_id', $user->id)->first();

            if ($madrasah) {
                $totalSiswa = Siswa::where('madrasah_id', $madrasah->id)->count();
                $totalKelas = Kelas::where('madrasah_id', $madrasah->id)->count();

                // Kehadiran bulan ini
                $startOfMonth = Carbon::now()->startOfMonth();
                $endOfMonth = Carbon::now()->endOfMonth();

                $presensiBulanIni = DetailPresensiMadrasah::whereHas('presensiMadrasah', function ($query) use ($madrasah, $startOfMonth, $endOfMonth) {
                    $query->whereHas('kelas', function ($q) use ($madrasah) {
                        $q->where('madrasah_id', $madrasah->id);
                    })->whereBetween('tanggal', [$startOfMonth, $endOfMonth]);
                })->count();

                $hadirBulanIni = DetailPresensiMadrasah::whereHas('presensiMadrasah', function ($query) use ($madrasah, $startOfMonth, $endOfMonth) {
                    $query->whereHas('kelas', function ($q) use ($madrasah) {
                        $q->where('madrasah_id', $madrasah->id);
                    })->whereBetween('tanggal', [$startOfMonth, $endOfMonth]);
                })->where('status', 'hadir')->count();

                $persentaseHadirBulanIni = $presensiBulanIni > 0 ? round(($hadirBulanIni / $presensiBulanIni) * 100, 1) : 0;

                $stats[] = Stat::make('Madrasah Anda', $madrasah->nama)
                    ->description('Total '.$totalSiswa.' siswa, '.$totalKelas.' kelas')
                    ->descriptionIcon('heroicon-o-building-office-2')
                    ->color('primary');

                $stats[] = Stat::make('Total Siswa', number_format($totalSiswa))
                    ->description('Siswa aktif')
                    ->descriptionIcon('heroicon-o-user-group')
                    ->color('success');

                $stats[] = Stat::make('Kehadiran Bulan Ini', $persentaseHadirBulanIni.'%')
                    ->description('Rata-rata kehadiran madrasah')
                    ->descriptionIcon('heroicon-o-chart-bar')
                    ->color($persentaseHadirBulanIni >= 80 ? 'success' : ($persentaseHadirBulanIni >= 60 ? 'warning' : 'danger'))
                    ->chart([70, 75, 78, 80, 82, $persentaseHadirBulanIni]);
            } else {
                $stats[] = Stat::make('Info', 'Anda belum ditugaskan di madrasah manapun')
                    ->color('gray');
            }
        }

        // Stats untuk Ortu (Orang Tua)
        if ($user->hasRole(['ortu', 'Ortu'])) {
            // Cari siswa yang merupakan anak dari user ini
            $siswa = $user->siswa;

            if ($siswa) {
                $today = Carbon::today();
                $startOfMonth = Carbon::now()->startOfMonth();
                $endOfMonth = Carbon::now()->endOfMonth();

                // Kehadiran Madrasah Hari Ini
                $presensiMadrasahToday = DetailPresensiMadrasah::where('siswa_id', $siswa->id)
                    ->whereHas('presensiMadrasah', function ($query) use ($today) {
                        $query->where('tanggal', $today);
                    })->count();

                $hadirMadrasahToday = DetailPresensiMadrasah::where('siswa_id', $siswa->id)
                    ->whereHas('presensiMadrasah', function ($query) use ($today) {
                        $query->where('tanggal', $today);
                    })
                    ->where('status', 'hadir')
                    ->count();

                $persentaseMadrasahToday = $presensiMadrasahToday > 0 ? round(($hadirMadrasahToday / $presensiMadrasahToday) * 100, 1) : 0;

                // Kehadiran Madrasah Bulan Ini
                $presensiMadrasahBulanIni = DetailPresensiMadrasah::where('siswa_id', $siswa->id)
                    ->whereHas('presensiMadrasah', function ($query) use ($startOfMonth, $endOfMonth) {
                        $query->whereBetween('tanggal', [$startOfMonth, $endOfMonth]);
                    })->count();

                $hadirMadrasahBulanIni = DetailPresensiMadrasah::where('siswa_id', $siswa->id)
                    ->whereHas('presensiMadrasah', function ($query) use ($startOfMonth, $endOfMonth) {
                        $query->whereBetween('tanggal', [$startOfMonth, $endOfMonth]);
                    })
                    ->where('status', 'hadir')
                    ->count();

                $persentaseMadrasahBulanIni = $presensiMadrasahBulanIni > 0 ? round(($hadirMadrasahBulanIni / $presensiMadrasahBulanIni) * 100, 1) : 0;

                // // Kehadiran Asrama (jika siswa tinggal di asrama)
                // $presensiAsramaBulanIni = 0;
                // $hadirAsramaBulanIni = 0;
                // $persentaseAsramaBulanIni = 0;

                // if ($siswa->kamar_id) {
                //     $presensiAsramaBulanIni = DetailPresensiAsrama::where('siswa_id', $siswa->id)
                //         ->whereHas('presensiAsrama', function ($query) use ($startOfMonth, $endOfMonth) {
                //             $query->whereBetween('tanggal', [$startOfMonth, $endOfMonth]);
                //         })->count();

                //     $hadirAsramaBulanIni = DetailPresensiAsrama::where('siswa_id', $siswa->id)
                //         ->whereHas('presensiAsrama', function ($query) use ($startOfMonth, $endOfMonth) {
                //             $query->whereBetween('tanggal', [$startOfMonth, $endOfMonth]);
                //         })
                //         ->where('status', 'hadir')
                //         ->count();

                //     $persentaseAsramaBulanIni = $presensiAsramaBulanIni > 0 ? round(($hadirAsramaBulanIni / $presensiAsramaBulanIni) * 100, 1) : 0;
                // }

                // Total Alpha Bulan Ini
                $alphaMadrasah = DetailPresensiMadrasah::where('siswa_id', $siswa->id)
                    ->whereHas('presensiMadrasah', function ($query) use ($startOfMonth, $endOfMonth) {
                        $query->whereBetween('tanggal', [$startOfMonth, $endOfMonth]);
                    })
                    ->where('status', 'alpha')
                    ->count();

                $alphaAsrama = DetailPresensiAsrama::where('siswa_id', $siswa->id)
                    ->whereHas('presensiAsrama', function ($query) use ($startOfMonth, $endOfMonth) {
                        $query->whereBetween('tanggal', [$startOfMonth, $endOfMonth]);
                    })
                    ->where('status', 'alpha')
                    ->count();

                $totalAlpha = $alphaMadrasah + $alphaAsrama;

                $stats[] = Stat::make('Anak Anda', $siswa->nama)
                    ->description('Kelas: '.($siswa->kelas->nama ?? '-').' | NISN: '.$siswa->nisn)
                    ->descriptionIcon('heroicon-o-user')
                    ->color('primary');

                // $stats[] = Stat::make('Kehadiran Sekolah Hari Ini', $persentaseMadrasahToday.'%')
                //     ->description($hadirMadrasahToday.' dari '.$presensiMadrasahToday.' kegiatan hadir')
                //     ->descriptionIcon('heroicon-o-academic-cap')
                //     ->color($persentaseMadrasahToday >= 80 ? 'success' : ($persentaseMadrasahToday >= 60 ? 'warning' : 'danger'));

                $stats[] = Stat::make('Kehadiran Kegiatan Sekolah Bulan Ini', $persentaseMadrasahBulanIni.'%')
                    ->description('Rata-rata kehadiran di sekolah')
                    ->descriptionIcon('heroicon-o-chart-bar')
                    ->color($persentaseMadrasahBulanIni >= 80 ? 'success' : ($persentaseMadrasahBulanIni >= 60 ? 'warning' : 'danger'))
                    ->chart([70, 75, 78, 80, 82, $persentaseMadrasahBulanIni]);

                // if ($siswa->kamar_id) {
                //     $stats[] = Stat::make('Kehadiran Asrama Bulan Ini', $persentaseAsramaBulanIni.'%')
                //         ->description('Rata-rata kehadiran di asrama')
                //         ->descriptionIcon('heroicon-o-home')
                //         ->color($persentaseAsramaBulanIni >= 80 ? 'success' : ($persentaseAsramaBulanIni >= 60 ? 'warning' : 'danger'))
                //         ->chart([70, 75, 78, 80, 82, $persentaseAsramaBulanIni]);
                // }

                $stats[] = Stat::make('Total Ketidakhadiran kegiatan', $totalAlpha.'x')
                    ->description('Alpha bulan ini (Sekolah: '.$alphaMadrasah.', Asrama: '.$alphaAsrama.')')
                    ->descriptionIcon('heroicon-o-exclamation-triangle')
                    ->color($totalAlpha == 0 ? 'success' : ($totalAlpha <= 3 ? 'warning' : 'danger'));
            } else {
                $stats[] = Stat::make('Info', 'Data siswa tidak ditemukan')
                    ->description('Silakan hubungi administrator')
                    ->color('gray');
            }
        }

        return $stats;
    }
}
