<?php

namespace App\Filament\Widgets;

use App\Models\JenisPresensiNonAsrama;
use App\Models\Kelas;
use App\Models\PresensiNonAsrama;
use App\Models\TahunAjaran;
use BezhanSalleh\FilamentShield\Traits\HasWidgetShield;
use Carbon\Carbon;
use Filament\Widgets\Concerns\InteractsWithPageFilters;
use Filament\Widgets\StatsOverviewWidget as BaseWidget;
use Filament\Widgets\StatsOverviewWidget\Stat;
use Illuminate\Support\Facades\Auth;

class PresensiNonAsramaWidget extends BaseWidget
{
    use \App\Traits\HasStrictWidgetShield, InteractsWithPageFilters;

    protected static ?int $sort = 10;

    protected function getStats(): array
    {
        $user = Auth::user();
        $kelasId = null;

        // If user is a walikelas, find their class
        if ($user->hasRole('walikelas')) {
            // Use global scope - it will automatically filter by madrasah
            $kelas = Kelas::where('wali_kelas_id', $user->id)->first();
            $kelasId = $kelas?->id;
        }
        // If user is Ortu, find their child's class
        elseif ($user->hasRole(['ortu', 'Ortu'])) {
            $siswa = $user->siswa;
            $kelasId = $siswa?->kelas_id;
        } else {
            // For other roles, use the page filter or the first class as a default
            // Global scope will automatically filter by madrasah
            $kelasId = $this->filters['kelas_id'] ?? Kelas::first()?->id;
        }

        if (! $kelasId) {
            $message = 'Belum ada data kelas.';
            if ($user->hasRole('walikelas')) {
                $message = 'Anda tidak ditugaskan di kelas manapun.';
            }
            if ($user->hasRole(['ortu', 'Ortu'])) {
                $message = 'Anda tidak memiliki siswa yang terdaftar.';
            }

            return [
                Stat::make('Info', $message)
                    ->color('gray'),
            ];
        }

        $today = Carbon::today();
        $tahunAjaran = TahunAjaran::where('is_aktif', true)->first();

        if (! $tahunAjaran) {
            return [
                Stat::make('Error', 'Tahun Ajaran Aktif tidak ditemukan')->color('danger'),
            ];
        }

        $jenisPresensi = JenisPresensiNonAsrama::where('is_active', true)->get();
        $stats = [];

        foreach ($jenisPresensi as $jenis) {
            $presensi = PresensiNonAsrama::where('tanggal', $today)
                ->where('kelas_id', $kelasId)
                ->where('jenis_presensi_non_asrama_id', $jenis->id)
                ->first();

            // Status Logic
            $status = $presensi ? 'Sudah Input' : 'Belum Input';
            $color = $presensi ? 'success' : 'danger';
            $icon = $presensi ? 'heroicon-o-check-circle' : 'heroicon-o-x-circle';

            // Description Logic
            if ($presensi) {
                $description = 'Terkunci: '.($presensi->is_locked ? 'Ya' : 'Tidak');
            } else {
                $description = 'Klik untuk input';
            }

            // URL Logic: Link to Custom Input Page
            // Custom Standalone Page Route
            $url = \App\Filament\Pages\InputPresensiNonAsrama::getUrl([
                'kelas_id' => $kelasId,
                'jenis_presensi_id' => $jenis->id,
                'tanggal' => $today->toDateString(),
            ]);

            $stat = Stat::make($jenis->nama, $status)
                ->description($description)
                ->descriptionIcon($icon)
                ->color($color)
                ->url($url);

            $stats[] = $stat;
        }

        return $stats;
    }
}
