<?php

namespace App\Filament\Widgets;

use App\Models\DetailPresensiAsrama;
use App\Models\DetailPresensiMadrasah;
use App\Models\Kamar;
use App\Models\Kelas;
use App\Models\Madrasah;
use App\Models\Siswa;
use App\Models\TahunAjaran;
use Carbon\Carbon;
use Filament\Widgets\StatsOverviewWidget as BaseWidget;
use Filament\Widgets\StatsOverviewWidget\Stat;
use Illuminate\Support\Facades\Auth;

class DashboardOverviewWidget extends BaseWidget
{
    use \App\Traits\HasStrictWidgetShield;

    protected static ?int $sort = 0;

    protected function getStats(): array
    {
        $user = Auth::user();
        $tahunAjaran = TahunAjaran::where('is_aktif', true)->first();

        // Total Siswa
        $totalSiswa = Siswa::count();
        $siswaThisMonth = Siswa::whereMonth('created_at', Carbon::now()->month)->count();
        $siswaGrowth = $siswaThisMonth > 0 ? "+{$siswaThisMonth} bulan ini" : 'Tidak ada siswa baru';

        // Total Madrasah
        $totalMadrasah = Madrasah::count();

        // Total Kelas
        $totalKelas = Kelas::count();

        // Total Kamar
        $totalKamar = Kamar::count();

        // Tingkat Kehadiran Rata-rata (30 hari terakhir)
        $startDate = Carbon::now()->subDays(30);
        $endDate = Carbon::now();

        $totalPresensiMadrasah = DetailPresensiMadrasah::whereHas('presensiMadrasah', function ($query) use ($startDate, $endDate) {
            $query->whereBetween('tanggal', [$startDate, $endDate]);
        })->count();

        $hadirMadrasah = DetailPresensiMadrasah::whereHas('presensiMadrasah', function ($query) use ($startDate, $endDate) {
            $query->whereBetween('tanggal', [$startDate, $endDate]);
        })->where('status', 'hadir')->count();

        $totalPresensiAsrama = DetailPresensiAsrama::whereHas('presensiAsrama', function ($query) use ($startDate, $endDate) {
            $query->whereBetween('tanggal', [$startDate, $endDate]);
        })->count();

        $hadirAsrama = DetailPresensiAsrama::whereHas('presensiAsrama', function ($query) use ($startDate, $endDate) {
            $query->whereBetween('tanggal', [$startDate, $endDate]);
        })->where('status', 'hadir')->count();

        $totalPresensi = $totalPresensiMadrasah + $totalPresensiAsrama;
        $totalHadir = $hadirMadrasah + $hadirAsrama;

        $tingkatKehadiran = $totalPresensi > 0 ? round(($totalHadir / $totalPresensi) * 100, 1) : 0;
        $kehadiranColor = $tingkatKehadiran >= 80 ? 'success' : ($tingkatKehadiran >= 60 ? 'warning' : 'danger');

        return [
            Stat::make('Total Siswa', number_format($totalSiswa))
                ->description($siswaGrowth)
                ->descriptionIcon('heroicon-o-user-group')
                ->color('primary')
                ->chart([7, 12, 15, 18, 22, 25, $totalSiswa]),

            Stat::make('Total Madrasah', number_format($totalMadrasah))
                ->description('Madrasah terdaftar')
                ->descriptionIcon('heroicon-o-building-office-2')
                ->color('success'),

            Stat::make('Total Kelas', number_format($totalKelas))
                ->description('Kelas aktif')
                ->descriptionIcon('heroicon-o-academic-cap')
                ->color('info'),

            Stat::make('Total Kamar', number_format($totalKamar))
                ->description('Kamar asrama')
                ->descriptionIcon('heroicon-o-home')
                ->color('warning'),

            Stat::make('Tingkat Kehadiran', $tingkatKehadiran.'%')
                ->description('Rata-rata 30 hari terakhir')
                ->descriptionIcon('heroicon-o-chart-bar')
                ->color($kehadiranColor)
                ->chart([65, 70, 75, 78, 80, 82, $tingkatKehadiran]),
        ];
    }
}
