<?php

namespace App\Filament\Resources;

use App\Filament\Exports\UserExporter;
use App\Filament\Imports\UserImporter;
use App\Filament\Resources\UserResource\Pages;
use App\Models\User;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Form;
use Filament\Infolists\Components\Section as InfolistSection;
use Filament\Infolists\Components\TextEntry;
use Filament\Infolists\Infolist;
use Filament\Resources\Resource;
use Filament\Support\Enums\FontWeight;
use Filament\Tables;
use Filament\Tables\Actions\Action;
use Filament\Tables\Actions\ExportAction;
use Filament\Tables\Actions\ExportBulkAction;
use Filament\Tables\Actions\ImportAction;
use Filament\Tables\Filters\SelectFilter;
use Filament\Tables\Table;
use STS\FilamentImpersonate\Tables\Actions\Impersonate;

class UserResource extends Resource
{
    public static function getNavigationBadge(): ?string
    {
        return static::getModel()::count();
    }

    protected static ?string $model = User::class;

    protected static ?string $navigationIcon = 'heroicon-o-users';

    protected static ?string $pluralModelLabel = 'User';

    protected static ?string $navigationGroup = 'Settings';

    protected static ?int $navigationSort = 2;

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Section::make(
                    'User Information'
                )->schema([
                    TextInput::make('name')
                        ->required(),
                    TextInput::make('email')
                        ->required(),
                    TextInput::make('password')
                        ->required(),
                    Select::make('madrasah_id')
                        ->relationship('madrasah', 'nama')
                        ->searchable()
                        ->preload()
                        ->label('Madrasah')
                        ->helperText('Pilih madrasah untuk user ini (opsional untuk super admin/pimpinan)'),
                    Select::make('roles')
                        ->relationship('roles', 'name')
                        ->multiple()
                        ->searchable()
                        ->preload()
                        ->label('Role')
                        ->reactive()
                        ->afterStateUpdated(fn ($state, callable $set) => $set('siswa_id', null)),
                    Select::make('siswa_id')
                        ->label('Link ke Siswa (untuk Ortu)')
                        ->options(function ($get, $record) {
                            $query = \App\Models\Siswa::query();

                            // Filter by madrasah if selected
                            if ($madrasahId = $get('madrasah_id')) {
                                $query->whereHas('kelas', fn ($q) => $q->where('madrasah_id', $madrasahId));
                            }

                            // Only show siswa without user_id OR current user's siswa (for edit)
                            $query->where(function ($q) use ($record) {
                                $q->whereNull('user_id');
                                if ($record) {
                                    $q->orWhere('user_id', $record->id);
                                }
                            });

                            return $query->pluck('nama', 'id');
                        })
                        ->searchable()
                        ->helperText('Hanya siswa yang belum ter-link ke user lain yang akan muncul')
                        ->visible(fn ($get) => in_array('ortu', $get('roles') ?? []))
                        ->reactive()
                        ->afterStateUpdated(function ($state, $get, $set) {
                            if ($state) {
                                $siswa = \App\Models\Siswa::find($state);
                                if ($siswa && ! $get('madrasah_id')) {
                                    $set('madrasah_id', $siswa->kelas->madrasah_id);
                                }
                            }
                        }),
                ]),
            ]);
    }

    public static function canCreate(): bool
    {
        return true;
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\Layout\Split::make([
                    Tables\Columns\ImageColumn::make('avatar_url')
                        ->searchable()
                        ->circular()
                        ->grow(false)
                        ->getStateUsing(fn ($record) => $record->avatar_url
                            ? $record->avatar_url
                            : 'https://ui-avatars.com/api/?name='.urlencode($record->name)),
                    Tables\Columns\TextColumn::make('name')
                        ->searchable()
                        ->weight(FontWeight::Bold),
                    Tables\Columns\Layout\Stack::make([
                        Tables\Columns\TextColumn::make('roles.name')
                            ->searchable()
                            ->icon('heroicon-o-shield-check')
                            ->grow(false),
                        Tables\Columns\TextColumn::make('email')
                            ->icon('heroicon-m-envelope')
                            ->searchable()
                            ->grow(false),
                    ])->alignStart()->visibleFrom('lg')->space(1),
                ]),
            ])
            ->filters([
                //
                SelectFilter::make('roles')
                    ->relationship('roles', 'name')
                    ->multiple()
                    ->preload(),
            ])
            ->actions([
                Tables\Actions\ViewAction::make(),
                Tables\Actions\EditAction::make(),
                Action::make('Set Role')
                    ->icon('heroicon-m-adjustments-vertical')
                    ->form([
                        Select::make('role')
                            ->relationship('roles', 'name')
                            ->multiple()
                            ->required()
                            ->searchable()
                            ->preload()
                            ->optionsLimit(10)
                            ->getOptionLabelFromRecordUsing(fn ($record) => $record->name),
                    ]),
                // Impersonate::make(),
                Tables\Actions\DeleteAction::make(),
            ])
            ->headerActions([
                ExportAction::make()
                    ->exporter(UserExporter::class),
                ImportAction::make()
                    ->importer(UserImporter::class),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
                ExportBulkAction::make()
                    ->exporter(UserExporter::class),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListUsers::route('/'),
            'create' => Pages\CreateUser::route('/create'),
            // 'view' => Pages\ViewUser::route('/{record}'),
            'edit' => Pages\EditUser::route('/{record}/edit'),
        ];
    }

    public static function infolist(Infolist $infolist): Infolist
    {
        return $infolist
            ->schema([
                InfolistSection::make('User Information')->schema([
                    TextEntry::make('name'),
                    TextEntry::make('email'),
                ]),
            ]);
    }
}
