<?php

namespace App\Filament\Resources;

use App\Exports\SiswaExcelExport;
use App\Exports\SiswaTemplateExport;
use App\Filament\Resources\SiswaResource\Pages;
use App\Imports\SiswaImport;
use App\Models\Siswa;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Notifications\Notification;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\Auth;
use Maatwebsite\Excel\Facades\Excel;

class SiswaResource extends Resource
{
    protected static ?string $model = Siswa::class;

    protected static ?string $navigationIcon = 'heroicon-o-user-group';

    protected static ?string $pluralModelLabel = 'Siswa';

    protected static ?string $navigationGroup = 'Akademik';

    protected static ?int $navigationSort = 22;

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\Grid::make()
                    ->columns(['default' => 1, 'md' => 2])
                    ->schema([
                        Forms\Components\Group::make()
                            ->schema([
                                Forms\Components\Section::make('Identitas Siswa')
                                    ->description('Data pribadi dan kelahiran')
                                    ->columns(2)
                                    ->schema([
                                        Forms\Components\TextInput::make('nisn')
                                            ->required()
                                            ->maxLength(20),
                                        Forms\Components\TextInput::make('nis')
                                            ->maxLength(20)
                                            ->default(null),
                                        Forms\Components\TextInput::make('nik')
                                            ->maxLength(20)
                                            ->default(null),
                                        Forms\Components\TextInput::make('nama')
                                            ->required()
                                            ->maxLength(255)
                                            ->columnSpan(2),
                                        Forms\Components\Select::make('jenis_kelamin')
                                            ->required()
                                            ->options([
                                                'L' => 'Laki-laki',
                                                'P' => 'Perempuan',
                                            ])
                                            ->placeholder('Pilih Jenis Kelamin'),
                                        Forms\Components\TextInput::make('tempat_lahir')
                                            ->maxLength(255)
                                            ->default(null),
                                        Forms\Components\DatePicker::make('tanggal_lahir')
                                            ->displayFormat('d/m/Y'),
                                    ]),
                                Forms\Components\Section::make('Data Ayah')
                                    ->description('Informasi ayah')
                                    ->columns(2)
                                    ->schema([
                                        Forms\Components\TextInput::make('nama_ayah')
                                            ->maxLength(255)
                                            ->default(null),
                                        Forms\Components\TextInput::make('nik_ayah')
                                            ->maxLength(20)
                                            ->default(null),
                                        Forms\Components\TextInput::make('pendidikan_ayah')
                                            ->maxLength(255)
                                            ->default(null),
                                        Forms\Components\TextInput::make('pekerjaan_ayah')
                                            ->maxLength(255)
                                            ->default(null),
                                    ]), Forms\Components\Section::make('Data Ibu')
                                    ->description('Informasi ibu')
                                    ->columns(2)
                                    ->schema([
                                        Forms\Components\TextInput::make('nama_ibu')
                                            ->maxLength(255)
                                            ->default(null),
                                        Forms\Components\TextInput::make('nik_ibu')
                                            ->maxLength(20)
                                            ->default(null),
                                        Forms\Components\TextInput::make('pendidikan_ibu')
                                            ->maxLength(255)
                                            ->default(null),
                                        Forms\Components\TextInput::make('pekerjaan_ibu')
                                            ->maxLength(255)
                                            ->default(null),
                                    ]),
                                Forms\Components\Section::make('Kontak Orang Tua')
                                    ->schema([
                                        Forms\Components\TextInput::make('no_hp_ortu')
                                            ->maxLength(20)
                                            ->default(null)
                                            ->helperText('Opsional'),
                                    ]),

                            ])
                            ->columnSpan(1),
                        Forms\Components\Group::make()
                            ->schema([
                                Forms\Components\Section::make('Akademik & Asrama')
                                    ->description('Informasi sekolah dan kamar')
                                    ->columns(2)
                                    ->schema([
                                        Forms\Components\Select::make('madrasah_id')
                                            ->relationship('madrasah', 'nama')
                                            ->searchable()
                                            ->preload()
                                            ->placeholder('Pilih Madrasah')
                                            ->required(),
                                        Forms\Components\Select::make('kelas_id')
                                            ->relationship('kelas', 'nama')
                                            ->searchable()
                                            ->preload()
                                            ->placeholder('Pilih Kelas')
                                            ->required(),
                                        Forms\Components\Select::make('status_tempat_tinggal')
                                            ->required()
                                            ->options([
                                                'asrama' => 'Asrama',
                                                'non_asrama' => 'Non Asrama',
                                            ])
                                            ->placeholder('Pilih Status Tinggal'),
                                        Forms\Components\Select::make('kamar_id')
                                            ->relationship('kamar', 'nama_kamar')
                                            ->searchable()
                                            ->preload()
                                            ->placeholder('Pilih Kamar')
                                            ->default(null),
                                        Forms\Components\TextInput::make('no_kk')
                                            ->maxLength(20)
                                            ->default(null)
                                            ->helperText('Opsional'),
                                        Forms\Components\Select::make('user_id')
                                            ->label('Akun Orang Tua')
                                            ->placeholder('Pilih akun orang tua')
                                            ->relationship('user', 'name')
                                            ->default(null),
                                    ]),

                                Forms\Components\Section::make('Alamat')
                                    ->description('Alamat lengkap')
                                    ->columns(2)
                                    ->schema([
                                        Forms\Components\Textarea::make('alamat')
                                            ->rows(3)
                                            ->columnSpanFull(),
                                        Forms\Components\TextInput::make('provinsi')
                                            ->maxLength(255)
                                            ->default(null),
                                        Forms\Components\TextInput::make('kabupaten_kota')
                                            ->maxLength(255)
                                            ->default(null),
                                        Forms\Components\TextInput::make('kecamatan')
                                            ->maxLength(255)
                                            ->default(null),
                                        Forms\Components\TextInput::make('desa_kelurahan')
                                            ->maxLength(255)
                                            ->default(null),
                                        Forms\Components\TextInput::make('kode_pos')
                                            ->maxLength(10)
                                            ->default(null),
                                    ]),
                            ])
                            ->columnSpan(1),
                    ]),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->modifyQueryUsing(fn (Builder $query) => $query->with(['madrasah', 'kelas', 'kamar']))
            ->columns([
                Tables\Columns\TextColumn::make('kelas.nama')
                    ->numeric()
                    ->sortable(),
                Tables\Columns\TextColumn::make('nisn')
                    ->searchable(),
                Tables\Columns\TextColumn::make('nama')
                    ->searchable()->sortable(),
                Tables\Columns\TextColumn::make('jenis_kelamin'),
                Tables\Columns\TextColumn::make('tempat_lahir')
                    ->searchable(),
                Tables\Columns\TextColumn::make('tanggal_lahir')
                    ->date()
                    ->sortable(),
                Tables\Columns\TextColumn::make('status_tempat_tinggal'),
                Tables\Columns\TextColumn::make('kamar.nama_kamar')
                    ->numeric()
                    ->sortable(),
                Tables\Columns\TextColumn::make('no_hp_ortu')
                    ->searchable(),

            ])
            ->filters([
                //
            ])
            ->headerActions([
                Tables\Actions\Action::make('import')
                    ->label('Import Siswa')
                    ->icon('heroicon-o-arrow-up-tray')
                    ->color('primary')
                    ->form([
                        Forms\Components\FileUpload::make('file')
                            ->label('File Excel')
                            ->acceptedFileTypes([
                                'application/vnd.ms-excel',
                                'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
                            ])
                            ->required()
                            ->helperText('Upload file Excel (.xlsx atau .xls)'),
                        Forms\Components\Actions::make([
                            Forms\Components\Actions\Action::make('downloadTemplate')
                                ->label('Download Template Excel')
                                ->icon('heroicon-o-arrow-down-tray')
                                ->color('success')
                                ->action(function () {
                                    $madrasahId = Auth::user()->madrasah_id;
                                    $filename = 'template-import-siswa.xlsx';

                                    return Excel::download(
                                        new SiswaTemplateExport($madrasahId),
                                        $filename
                                    );
                                }),
                        ])->fullWidth(),
                    ])
                    ->action(function (array $data, $livewire) {
                        try {
                            // madrasah_id akan diambil dari Excel, jadi tidak perlu dari user
                            $import = new SiswaImport(null);

                            // Ambil file dari Livewire temporary storage
                            $fileKey = $data['file'];

                            // Coba beberapa kemungkinan path
                            $possiblePaths = [
                                storage_path('app/livewire-tmp/'.$fileKey),
                                storage_path('app/public/'.$fileKey),
                                storage_path('app/'.$fileKey),
                            ];

                            $filePath = null;
                            foreach ($possiblePaths as $path) {
                                if (file_exists($path)) {
                                    $filePath = $path;
                                    break;
                                }
                            }

                            // Jika masih tidak ketemu, coba cari di semua subdirectory livewire-tmp
                            if (! $filePath) {
                                $livewireTmpDir = storage_path('app/livewire-tmp');
                                if (is_dir($livewireTmpDir)) {
                                    // Cari file yang mengandung key atau nama file
                                    $iterator = new \RecursiveIteratorIterator(
                                        new \RecursiveDirectoryIterator($livewireTmpDir, \RecursiveDirectoryIterator::SKIP_DOTS)
                                    );

                                    foreach ($iterator as $file) {
                                        if ($file->isFile() && (
                                            str_contains($file->getFilename(), $fileKey) ||
                                            str_contains($file->getPathname(), $fileKey)
                                        )) {
                                            $filePath = $file->getPathname();
                                            break;
                                        }
                                    }
                                }
                            }

                            if (! $filePath || ! file_exists($filePath)) {
                                throw new \Exception('File tidak ditemukan. File key: '.$fileKey.'. Pastikan file berhasil diupload.');
                            }

                            Excel::import($import, $filePath);

                            // Hapus file temporary setelah import
                            if (file_exists($filePath)) {
                                @unlink($filePath);
                            }

                            $successCount = $import->getSuccessCount();
                            $failedCount = $import->getFailedCount();
                            $errors = $import->getErrors();

                            if ($failedCount > 0) {
                                $errorMessage = "Berhasil: {$successCount}, Gagal: {$failedCount}\n\n";
                                $errorMessage .= "Detail Error:\n".implode("\n", array_slice($errors, 0, 10));

                                if (count($errors) > 10) {
                                    $errorMessage .= "\n... dan ".(count($errors) - 10).' error lainnya';
                                }

                                Notification::make()
                                    ->title('Import selesai dengan beberapa error')
                                    ->body($errorMessage)
                                    ->warning()
                                    ->duration(10000)
                                    ->send();
                            } else {
                                Notification::make()
                                    ->title('Import berhasil!')
                                    ->body("Berhasil mengimport {$successCount} siswa")
                                    ->success()
                                    ->send();
                            }
                        } catch (\Exception $e) {
                            Notification::make()
                                ->title('Import gagal!')
                                ->body($e->getMessage())
                                ->danger()
                                ->send();
                        }
                    })
                    ->modalWidth('lg')
                    ->modalHeading('Import Data Siswa')
                    ->modalDescription('Upload file Excel untuk mengimport data siswa. Pastikan format sesuai dengan template yang disediakan.')
                    ->modalSubmitActionLabel('Import')
                    ->modalCancelActionLabel('Batal'),
                Tables\Actions\Action::make('export')
                    ->label('Export')
                    ->icon('heroicon-o-arrow-down-tray')
                    ->color('success')
                    ->action(function ($livewire) {
                        $records = $livewire->getFilteredTableQuery()
                            ->with(['madrasah', 'kelas', 'kamar', 'user'])
                            ->get();

                        $filename = 'siswa-export-'.now()->format('Y-m-d-His').'.xlsx';

                        Notification::make()
                            ->title('Export berhasil!')
                            ->success()
                            ->send();

                        return Excel::download(new SiswaExcelExport($records), $filename);
                    }),
            ])
            ->paginated([10, 25, 50, 100])
            ->actions([
                Tables\Actions\EditAction::make(),
                Tables\Actions\DeleteAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListSiswas::route('/'),
            'create' => Pages\CreateSiswa::route('/create'),
            'edit' => Pages\EditSiswa::route('/{record}/edit'),
        ];
    }
}
