<?php

namespace App\Filament\Resources\PresensiAsramaResource\RelationManagers;

use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\RelationManagers\RelationManager;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Filament\Notifications\Notification;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Support\Str;

class DetailPresensiAsramasRelationManager extends RelationManager
{
    protected static string $relationship = 'detailPresensiAsramas';

    protected static ?string $title = 'Detail Presensi Siswa';

    public function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\Select::make('siswa_id')
                    ->relationship('siswa', 'nama', fn (Builder $query, $livewire) => 
                        $query->whereHas('kamar', function ($q) use ($livewire) {
                            $q->where('id', $livewire->getOwnerRecord()->kamar_id);
                        })
                    )
                    ->searchable()
                    ->preload()
                    ->disabledOn('edit')
                    ->required()
                    ->label('Siswa'),
                Forms\Components\Select::make('status')
                    ->options(function ($livewire) {
                        $presensi = $livewire->getOwnerRecord();
                        // Load relationship if not loaded
                        if (!$presensi->relationLoaded('jenisPresensiAsrama')) {
                            $presensi->load('jenisPresensiAsrama');
                        }
                        
                        $isPuasa = Str::contains($presensi->jenisPresensiAsrama->nama ?? '', 'Puasa');

                        return [
                            'hadir' => $isPuasa ? 'Berpuasa' : 'Hadir',
                            'sakit' => 'Sakit',
                            'izin' => $isPuasa ? 'Berhalangan (Haid)' : 'Izin',
                            'alpha' => $isPuasa ? 'Tidak Berpuasa' : 'Alpha',
                        ];
                    })
                    ->required()
                    ->default('hadir')
                    ->label('Status Kehadiran'),
                Forms\Components\Textarea::make('keterangan')
                    ->rows(2)
                    ->columnSpanFull(),
            ]);
    }

    public function table(Table $table): Table
    {
        return $table
            ->recordTitleAttribute('status')
            ->columns([
                Tables\Columns\TextColumn::make('siswa.nisn')
                    ->label('NISN')
                    ->searchable()
                    ->sortable(),
                Tables\Columns\TextColumn::make('siswa.nama')
                    ->label('Nama Siswa')
                    ->searchable()
                    ->sortable(),
                Tables\Columns\TextColumn::make('status')
                    ->badge()
                    ->color(fn (?string $state): string => match ($state) {
                        'hadir' => 'success',
                        'sakit' => 'warning',
                        'izin' => 'info',
                        'alpha' => 'danger',
                        default => 'gray',
                    })
                    ->formatStateUsing(function (?string $state, $livewire): string {
                        if (!$state) return '-';
                        
                        $presensi = $livewire->getOwnerRecord();
                        if (!$presensi->relationLoaded('jenisPresensiAsrama')) {
                            $presensi->load('jenisPresensiAsrama');
                        }
                        $isPuasa = Str::contains($presensi->jenisPresensiAsrama->nama ?? '', 'Puasa');

                        if ($isPuasa) {
                            return match ($state) {
                                'hadir' => 'Berpuasa',
                                'alpha' => 'Tidak Berpuasa',
                                'izin' => 'Berhalangan (Haid)',
                                default => ucfirst($state),
                            };
                        }

                        return ucfirst($state);
                    })
                    ->sortable(),
                Tables\Columns\TextColumn::make('keterangan')
                    ->limit(30)
                    ->tooltip(fn (?string $state): ?string => $state),
            ])
            ->filters([
                Tables\Filters\SelectFilter::make('status')
                    ->options([
                        'hadir' => 'Hadir / Berpuasa',
                        'sakit' => 'Sakit',
                        'izin' => 'Izin / Berhalangan',
                        'alpha' => 'Alpha / Tidak Berpuasa',
                    ]),
            ])
            ->headerActions([
                Tables\Actions\CreateAction::make()
                    ->label('Tambah Manual')
                    ->visible(fn ($livewire) => ! $livewire->getOwnerRecord()->is_locked),
                Tables\Actions\Action::make('generate_siswa')
                    ->label('Generate Siswa')
                    ->icon('heroicon-o-users')
                    ->requiresConfirmation()
                    ->color('primary')
                    ->action(function ($livewire) {
                        $presensi = $livewire->getOwnerRecord();
                        $kamar = $presensi->kamar;
                        
                        if (! $kamar) {
                            Notification::make()
                                ->title('Kamar tidak ditemukan')
                                ->danger()
                                ->send();
                            return;
                        }

                        $siswas = $kamar->siswas;
                        $count = 0;

                        foreach ($siswas as $siswa) {
                            // Check if already exists
                            $exists = $presensi->detailPresensiAsramas()
                                ->where('siswa_id', $siswa->id)
                                ->exists();
                            
                            if (! $exists) {
                                $presensi->detailPresensiAsramas()->create([
                                    'siswa_id' => $siswa->id,
                                    'status' => 'hadir', // Default
                                ]);
                                $count++;
                            }
                        }
                        
                        Notification::make()
                            ->title("Berhasil generate $count siswa")
                            ->success()
                            ->send();
                    })
                    ->visible(fn ($livewire) => $livewire->getOwnerRecord()->detailPresensiAsramas()->count() === 0 && ! $livewire->getOwnerRecord()->is_locked),
            ])
            ->actions([
                Tables\Actions\EditAction::make()
                    ->visible(fn ($livewire) => ! $livewire->getOwnerRecord()->is_locked),
                Tables\Actions\DeleteAction::make()
                    ->visible(fn ($livewire) => ! $livewire->getOwnerRecord()->is_locked),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make()
                        ->visible(fn ($livewire) => ! $livewire->getOwnerRecord()->is_locked),
                    Tables\Actions\BulkAction::make('set_status')
                        ->label('Set Status')
                        ->icon('heroicon-o-pencil')
                        ->form(function ($livewire) {
                            $presensi = $livewire->getOwnerRecord();
                            if (!$presensi->relationLoaded('jenisPresensiAsrama')) {
                                $presensi->load('jenisPresensiAsrama');
                            }
                            $isPuasa = Str::contains($presensi->jenisPresensiAsrama->nama ?? '', 'Puasa');

                            return [
                                Forms\Components\Select::make('status')
                                    ->options([
                                        'hadir' => $isPuasa ? 'Berpuasa' : 'Hadir',
                                        'sakit' => 'Sakit',
                                        'izin' => $isPuasa ? 'Berhalangan (Haid)' : 'Izin',
                                        'alpha' => $isPuasa ? 'Tidak Berpuasa' : 'Alpha',
                                    ])
                                    ->required(),
                            ];
                        })
                        ->action(function (Collection $records, array $data) {
                            $records->each(function ($record) use ($data) {
                                $record->update(['status' => $data['status']]);
                            });
                            
                            Notification::make()
                                ->title('Status berhasil diperbarui')
                                ->success()
                                ->send();
                        })
                        ->deselectRecordsAfterCompletion()
                        ->visible(fn ($livewire) => ! $livewire->getOwnerRecord()->is_locked),
                ]),
            ]);
    }
}
