<?php

namespace App\Filament\Resources;

use App\Filament\Resources\PresensiAsramaResource\Pages;
use App\Filament\Resources\PresensiAsramaResource\RelationManagers;
use App\Models\PresensiAsrama;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Support\Facades\Auth;

class PresensiAsramaResource extends Resource
{
    protected static ?string $model = PresensiAsrama::class;

    protected static ?string $navigationIcon = 'heroicon-o-clipboard-document-check';

    protected static ?string $pluralModelLabel = 'Presensi Asrama';

    protected static ?string $navigationGroup = 'Kegiatan Pengasuhan';

    protected static ?int $navigationSort = 39;

    public static function getEloquentQuery(): \Illuminate\Database\Eloquent\Builder
    {
        $query = parent::getEloquentQuery();
        $user = Auth::user();

        if (! $user) {
            return $query;
        }

        // Super admin, pimpinan, operator can see all
        if ($user->hasRole(['super_admin', 'pimpinan', 'operator'])) {
            return $query;
        }

        // Get user's madrasah ID
        $madrasahId = $user->getMadrasahId();

        if (! $madrasahId) {
            return $query->whereRaw('1 = 0');
        }

        // Filter through kamar->madrasah relationship
        $query->whereHas('kamar', function ($q) use ($madrasahId) {
            $q->where('madrasah_id', $madrasahId);
        });

        // Wali kamar hanya bisa lihat presensi kamar yang dia pegang
        if ($user->hasRole('wali kamar')) {
            $kamar = \App\Models\Kamar::where('wali_kamar_id', $user->id)->first();
            if ($kamar) {
                $query->where('kamar_id', $kamar->id);
            } else {
                $query->whereRaw('1 = 0');
            }
        }

        return $query;
    }

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\Section::make('Informasi Presensi')
                    ->schema([
                        Forms\Components\DatePicker::make('tanggal')
                            ->required()
                            ->default(now())
                            ->displayFormat('d/m/Y')
                            ->label('Tanggal'),
                        Forms\Components\Select::make('tahun_ajaran_id')
                            ->relationship('tahunAjaran', 'nama')
                            ->required()
                            ->searchable()
                            ->preload()
                            ->label('Tahun Ajaran'),
                        Forms\Components\Select::make('jenis_presensi_asrama_id')
                            ->relationship('jenisPresensiAsrama', 'nama')
                            ->required()
                            ->searchable()
                            ->preload()
                            ->label('Jenis Presensi'),
                        Forms\Components\Select::make('kamar_id')
                            ->relationship('kamar', 'nama_kamar', function ($query) {
                                $user = Auth::user();
                                if ($user && $user->hasRole('wali kamar')) {
                                    $query->where('wali_kamar_id', $user->id);
                                }
                                return $query;
                            })
                            ->required()
                            ->searchable()
                            ->preload()
                            ->label('Kamar'),
                        Forms\Components\Select::make('user_id')
                            ->relationship('user', 'name')
                            ->required()
                            ->searchable()
                            ->preload()
                            ->default(Auth::id())
                            ->label('Petugas Input'),
                        Forms\Components\Toggle::make('is_locked')
                            ->label('Kunci Presensi')
                            ->helperText('Jika dikunci, presensi tidak bisa diedit lagi')
                            ->default(false),
                        Forms\Components\Textarea::make('keterangan')
                            ->rows(3)
                            ->columnSpanFull()
                            ->label('Keterangan'),
                    ])
                    ->columns(2),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->defaultSort('tanggal', 'desc')
            ->columns([
                Tables\Columns\TextColumn::make('tanggal')
                    ->date('d/m/Y')
                    ->sortable()
                    ->label('Tanggal'),
                Tables\Columns\TextColumn::make('kamar.nama_kamar')
                    ->searchable()
                    ->sortable()
                    ->label('Kamar'),
                Tables\Columns\TextColumn::make('jenisPresensiAsrama.nama')
                    ->searchable()
                    ->sortable()
                    ->label('Jenis Presensi'),
                Tables\Columns\TextColumn::make('jenisPresensiAsrama.kategori')
                    ->badge()
                    ->label('Kategori'),
                Tables\Columns\TextColumn::make('detail_presensi_asramas_count')
                    ->counts('detailPresensiAsramas')
                    ->label('Jml Siswa')
                    ->badge()
                    ->color('success'),
                Tables\Columns\IconColumn::make('is_locked')
                    ->boolean()
                    ->label('Terkunci')
                    ->trueIcon('heroicon-o-lock-closed')
                    ->falseIcon('heroicon-o-lock-open'),
                Tables\Columns\TextColumn::make('user.name')
                    ->label('Petugas')
                    ->toggleable(isToggledHiddenByDefault: true),
                Tables\Columns\TextColumn::make('created_at')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true)
                    ->label('Dibuat'),
            ])
            ->filters([
                Tables\Filters\SelectFilter::make('tahun_ajaran_id')
                    ->relationship('tahunAjaran', 'nama')
                    ->label('Tahun Ajaran'),
                Tables\Filters\SelectFilter::make('kamar_id')
                    ->relationship('kamar', 'nama_kamar', function ($query) {
                        $user = Auth::user();
                        if ($user && $user->hasRole('wali kamar')) {
                            $query->where('wali_kamar_id', $user->id);
                        }
                        return $query;
                    })
                    ->label('Kamar'),
                Tables\Filters\SelectFilter::make('jenis_presensi_asrama_id')
                    ->relationship('jenisPresensiAsrama', 'nama')
                    ->label('Jenis Presensi'),
                Tables\Filters\TernaryFilter::make('is_locked')
                    ->label('Status Kunci'),
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
                Tables\Actions\DeleteAction::make()
                    ->visible(fn ($record) => ! $record->is_locked),
                Tables\Actions\Action::make('toggle_lock')
                    ->label(fn ($record) => $record->is_locked ? 'Buka Kunci' : 'Kunci')
                    ->icon(fn ($record) => $record->is_locked ? 'heroicon-o-lock-open' : 'heroicon-o-lock-closed')
                    ->color(fn ($record) => $record->is_locked ? 'warning' : 'success')
                    ->requiresConfirmation()
                    ->action(fn ($record) => $record->update(['is_locked' => ! $record->is_locked])),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            RelationManagers\DetailPresensiAsramasRelationManager::class,
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListPresensiAsramas::route('/'),
            'create' => Pages\CreatePresensiAsrama::route('/create'),
            'edit' => Pages\EditPresensiAsrama::route('/{record}/edit'),
        ];
    }
}
