<?php

namespace App\Filament\Resources;

use App\Filament\Resources\LaporanPresensiAsramaResource\Pages;
use App\Models\DetailPresensiAsrama;
use App\Models\JenisPresensiAsrama;
use App\Models\LaporanPresensiAsrama;
use App\Models\Siswa;
use App\Models\TahunAjaran;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\Auth;

class LaporanPresensiAsramaResource extends Resource
{
    protected static ?string $model = LaporanPresensiAsrama::class;

    protected static ?string $navigationIcon = 'heroicon-o-document-chart-bar';

    protected static ?string $navigationLabel = ' Kegiatan Pengasuhan';

    protected static ?string $pluralModelLabel = ' Kegiatan Pengasuhan';

    protected static ?string $navigationGroup = 'Laporan';

    protected static ?int $navigationSort = 51;

    public static function form(Form $form): Form
    {
        return $form->schema([]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->query(fn () => self::getTableQuery())
            ->columns(self::getTableColumns())
            ->filters(self::getTableFilters())
            ->defaultPaginationPageOption(10)
            ->headerActions([
                Tables\Actions\Action::make('pdf')
                    ->label('Export PDF')
                    ->icon('heroicon-o-document-arrow-down')
                    ->form([
                        \Filament\Forms\Components\Select::make('tahun_ajaran_id')
                            ->label('Tahun Ajaran')
                            ->options(TahunAjaran::pluck('nama', 'id'))
                            ->default(TahunAjaran::where('is_aktif', true)->first()?->id)
                            ->required()
                            ->searchable(),
                        \Filament\Forms\Components\Select::make('madrasah_id')
                            ->label('Madrasah')
                            ->options(\App\Models\Madrasah::pluck('nama', 'id'))
                            ->live()
                            ->afterStateUpdated(fn (callable $set) => $set('kamar_id', null))
                            ->placeholder('Pilih Madrasah (Opsional - Kosongkan untuk semua)')
                            ->searchable(),

                        \Filament\Forms\Components\Select::make('kamar_id')
                            ->label('Kamar')
                            ->options(function (callable $get) {
                                $user = Auth::user();
                                $query = \App\Models\Kamar::query();

                                // Filter by madrasah if selected
                                if ($madrasahId = $get('madrasah_id')) {
                                    $query->where('madrasah_id', $madrasahId);
                                }

                                // Filter by wali kamar role
                                if ($user && $user->hasRole('wali kamar')) {
                                    $query->where('wali_kamar_id', $user->id);
                                }

                                return $query->pluck('nama_kamar', 'id');
                            })
                            ->placeholder('Pilih Kamar (Opsional - Kosongkan untuk semua)')
                            ->searchable(),

                        \Filament\Forms\Components\Select::make('jenis_presensi_asrama_id')
                            ->label('Jenis Presensi')
                            ->options(JenisPresensiAsrama::where('is_active', true)->pluck('nama', 'id'))
                            ->placeholder('Semua Jenis Presensi')
                            ->searchable(),

                        \Filament\Forms\Components\DatePicker::make('dari')
                            ->label('Dari Tanggal')
                            ->default(now()->startOfMonth()),

                        \Filament\Forms\Components\DatePicker::make('sampai')
                            ->label('Sampai Tanggal')
                            ->default(now()),
                    ])
                    ->modalHeading('Export Laporan PDF')
                    ->modalDescription('Pilih filter untuk export laporan. Kosongkan Madrasah/Kamar untuk export semua data.')
                    ->modalSubmitActionLabel('Export PDF')
                    ->modalWidth('2xl')
                    ->action(function (array $data) {
                        // Increase memory limit for PDF generation
                        ini_set('memory_limit', '3048M');
                        set_time_limit(300);

                        // Re-use logic from Export class but for direct View render
                        $export = new \App\Exports\LaporanAsramaExport($data, 'pdf');
                        $view = $export->view();

                        $pdf = \Barryvdh\DomPDF\Facade\Pdf::loadView(
                            $view->name(),
                            $view->getData()
                        )->setPaper('a4', 'landscape');

                        return response()->streamDownload(function () use ($pdf) {
                            echo $pdf->output();
                        }, 'Laporan-Kegiatan-Pengasuhan-'.date('Y-m-d').'.pdf');
                    }),
            ])
            ->actions([
                Tables\Actions\Action::make('lihat_keterangan')
                    ->label('Detail')
                    ->icon('heroicon-o-eye')
                    ->color('info')
                    ->modalHeading(fn ($record) => 'Detail Keterangan - ' . $record->nama)
                    ->modalDescription(fn ($record) => 'NISN: ' . $record->nisn . ' | Kamar: ' . ($record->kamar->nama_kamar ?? '-'))
                    ->modalContent(function ($record) {
                        // Get filters
                        $tahunAjaranId = request()->input('tableFilters.tahun_ajaran_id.tahun_ajaran_id');
                        $jenisPresensiId = request()->input('tableFilters.jenis_presensi_asrama_id.jenis_presensi_asrama_id');
                        $dari = request()->input('tableFilters.tanggal.dari');
                        $sampai = request()->input('tableFilters.tanggal.sampai');

                        // Get all keterangan
                        $query = DetailPresensiAsrama::where('siswa_id', $record->id)
                            ->whereIn('status', ['sakit', 'izin', 'alpha'])
                            ->whereNotNull('keterangan')
                            ->where('keterangan', '!=', '');

                        $query->whereHas('presensiAsrama', function ($q) use ($tahunAjaranId, $jenisPresensiId, $dari, $sampai) {
                            if ($tahunAjaranId) {
                                $q->where('tahun_ajaran_id', $tahunAjaranId);
                            }
                            if ($jenisPresensiId) {
                                $q->where('jenis_presensi_asrama_id', $jenisPresensiId);
                            }
                            if ($dari) {
                                $q->where('tanggal', '>=', $dari);
                            }
                            if ($sampai) {
                                $q->where('tanggal', '<=', $sampai);
                            }
                        });

                        $details = $query->with(['presensiAsrama' => function ($q) {
                            $q->select('id', 'tanggal', 'jenis_presensi_asrama_id')
                                ->with('jenisPresensiAsrama:id,nama');
                        }])
                            ->orderBy('created_at', 'desc')
                            ->get();

                        if ($details->isEmpty()) {
                            return view('filament.components.empty-state', [
                                'message' => 'Tidak ada keterangan untuk santri ini.'
                            ]);
                        }

                        return view('filament.components.keterangan-detail-asrama', [
                            'details' => $details
                        ]);
                    })
                    ->modalSubmitAction(false)
                    ->modalCancelActionLabel('Tutup')
                    ->visible(fn ($record) => DetailPresensiAsrama::where('siswa_id', $record->id)
                        ->whereIn('status', ['sakit', 'izin', 'alpha'])
                        ->whereNotNull('keterangan')
                        ->where('keterangan', '!=', '')
                        ->exists()
                    ),
            ])
            ->bulkActions([])
            ->recordAction('lihat_keterangan')
            ->recordUrl(null)
            ->defaultSort('nama');
    }

    protected static function getTableQuery(): Builder
    {
        $query = LaporanPresensiAsrama::query()
            ->select('siswas.*')
            ->whereHas('kamar');

        // Scoped access untuk wali kamar - hanya lihat siswa di kamarnya
        $user = Auth::user();

        if ($user) {
            // Super admin, pimpinan, operator can see all
            if (! $user->hasRole(['super_admin', 'pimpinan', 'operator'])) {
                // Get user's madrasah ID
                $madrasahId = $user->getMadrasahId();

                if ($madrasahId) {
                    // Filter through kamar->madrasah relationship
                    $query->whereHas('kamar', function ($q) use ($madrasahId) {
                        $q->where('madrasah_id', $madrasahId);
                    });
                } else {
                    // If user has no madrasah, return empty query
                    $query->whereRaw('1 = 0');
                }
            }

            // Wali kamar only sees students in their room
            if ($user->hasRole('wali kamar')) {
                $kamar = \App\Models\Kamar::where('wali_kamar_id', $user->id)->first();
                if ($kamar) {
                    $query->where('kamar_id', $kamar->id);
                } else {
                    // Jika wali kamar tidak punya kamar, return empty query
                    $query->whereRaw('1 = 0');
                }
            }
        }

        // Get filters
        $tahunAjaranId = request()->input('tableFilters.tahun_ajaran_id.value');
        $jenisPresensiId = request()->input('tableFilters.jenis_presensi_asrama_id.value');
        $dari = request()->input('tableFilters.tanggal.dari');
        $sampai = request()->input('tableFilters.tanggal.sampai');

        // Build subquery conditions
        $subqueryConditions = function ($q) use ($tahunAjaranId, $jenisPresensiId, $dari, $sampai) {
            $q->whereHas('presensiAsrama', function ($q) use ($tahunAjaranId, $jenisPresensiId, $dari, $sampai) {
                if ($tahunAjaranId) {
                    $q->where('tahun_ajaran_id', $tahunAjaranId);
                }
                if ($jenisPresensiId) {
                    $q->where('jenis_presensi_asrama_id', $jenisPresensiId);
                }
                if ($dari) {
                    $q->where('tanggal', '>=', $dari);
                }
                if ($sampai) {
                    $q->where('tanggal', '<=', $sampai);
                }
            });
        };

        // Add counts with optimized queries
        $query->withCount([
            'detailPresensiAsramas as total_hadir' => function ($q) use ($subqueryConditions) {
                $q->where('status', 'hadir');
                $subqueryConditions($q);
            },
            'detailPresensiAsramas as total_sakit' => function ($q) use ($subqueryConditions) {
                $q->where('status', 'sakit');
                $subqueryConditions($q);
            },
            'detailPresensiAsramas as total_izin' => function ($q) use ($subqueryConditions) {
                $q->where('status', 'izin');
                $subqueryConditions($q);
            },
            'detailPresensiAsramas as total_alpha' => function ($q) use ($subqueryConditions) {
                $q->where('status', 'alpha');
                $subqueryConditions($q);
            },
        ]);

        return $query;
    }

    protected static function getTableColumns(): array
    {
        return [
            Tables\Columns\TextColumn::make('nisn')
                ->label('NISN')
                ->searchable()
                ->sortable(),
            Tables\Columns\TextColumn::make('nama')
                ->label('Nama Santri')
                ->searchable()
                ->sortable(),
            Tables\Columns\TextColumn::make('kamar.nama_kamar')
                ->label('Kamar')
                ->sortable()
                ->searchable(),
            Tables\Columns\TextColumn::make('total_hadir')
                ->label('B / H')
                ->alignCenter()
                ->sortable(),
            Tables\Columns\TextColumn::make('total_sakit')
                ->label('S')
                ->alignCenter()
                ->sortable(),
            Tables\Columns\TextColumn::make('total_izin')
                ->label('B/I')
                ->alignCenter()
                ->sortable(),
            Tables\Columns\TextColumn::make('total_alpha')
                ->label('T/A')
                ->alignCenter()
                ->sortable(),
            Tables\Columns\TextColumn::make('persentase')
                ->label('Persentase Hadir')
                ->getStateUsing(function ($record) {
                    $total = $record->total_hadir + $record->total_sakit + $record->total_izin + $record->total_alpha;

                    return $total > 0 ? round(($record->total_hadir / $total) * 100, 2).'%' : '0%';
                })
                ->alignCenter()
                ->badge()
                ->color(function (string $state): string {
                    $percentage = (float) str_replace('%', '', $state);

                    return match (true) {
                        $percentage >= 90 => 'success',
                        $percentage >= 75 => 'warning',
                        default => 'danger',
                    };
                }),
            Tables\Columns\TextColumn::make('keterangan')
                ->label('Keterangan')
                ->getStateUsing(function ($record) {
                    // Get filters
                    $tahunAjaranId = request()->input('tableFilters.tahun_ajaran_id.tahun_ajaran_id');
                    $jenisPresensiId = request()->input('tableFilters.jenis_presensi_asrama_id.jenis_presensi_asrama_id');
                    $dari = request()->input('tableFilters.tanggal.dari');
                    $sampai = request()->input('tableFilters.tanggal.sampai');

                    // Get keterangan for non-hadir status
                    $query = DetailPresensiAsrama::where('siswa_id', $record->id)
                        ->whereIn('status', ['sakit', 'izin', 'alpha'])
                        ->whereNotNull('keterangan')
                        ->where('keterangan', '!=', '');

                    // Apply filters
                    $query->whereHas('presensiAsrama', function ($q) use ($tahunAjaranId, $jenisPresensiId, $dari, $sampai) {
                        if ($tahunAjaranId) {
                            $q->where('tahun_ajaran_id', $tahunAjaranId);
                        }
                        if ($jenisPresensiId) {
                            $q->where('jenis_presensi_asrama_id', $jenisPresensiId);
                        }
                        if ($dari) {
                            $q->where('tanggal', '>=', $dari);
                        }
                        if ($sampai) {
                            $q->where('tanggal', '<=', $sampai);
                        }
                    });

                    $details = $query->with('presensiAsrama:id,tanggal')
                        ->orderBy('created_at', 'desc')
                        ->get();

                    if ($details->isEmpty()) {
                        return '-';
                    }

                    // Group by status and create summary like PDF format
                    $grouped = [];
                    foreach ($details as $detail) {
                        $status = $detail->status;
                        if (!isset($grouped[$status])) {
                            $grouped[$status] = [];
                        }
                        $grouped[$status][] = $detail->keterangan;
                    }

                    $parts = [];
                    foreach ($grouped as $status => $keterangans) {
                        $statusLabel = strtoupper(substr($status, 0, 1)); // S, I, A
                        $count = count($keterangans);
                        // Show first keterangan as sample (30 chars)
                        $sample = mb_substr($keterangans[0], 0, 30);
                        if (mb_strlen($keterangans[0]) > 30) $sample .= '...';
                        $parts[] = "{$statusLabel}: {$sample}" . ($count > 1 ? " (+".($count-1).")" : "");
                    }

                    return implode('; ', $parts);
                })
                ->searchable(false)
                ->sortable(false)
                ->wrap(),
        ];
    }

    protected static function getTableFilters(): array
    {
        return [
            Tables\Filters\Filter::make('tahun_ajaran_id')
                ->form([
                    \Filament\Forms\Components\Select::make('tahun_ajaran_id')
                        ->label('Tahun Ajaran')
                        ->options(TahunAjaran::pluck('nama', 'id'))
                        ->default(TahunAjaran::where('is_aktif', true)->first()?->id),
                ])
                ->query(function (Builder $query, array $data): Builder {
                    // Filter akan ditangani di getAttendanceCount method
                    return $query;
                }),
            Tables\Filters\Filter::make('madrasah_id')
                ->form([
                    \Filament\Forms\Components\Select::make('madrasah_id')
                        ->label('Madrasah')
                        ->options(\App\Models\Madrasah::pluck('nama', 'id'))
                        ->live()
                        ->afterStateUpdated(fn (callable $set) => $set('kamar_id', null))
                        ->placeholder('Pilih Madrasah (Opsional - Kosongkan untuk semua)')
                        ->searchable(),
                    \Filament\Forms\Components\Select::make('kamar_id')
                        ->label('Kamar')
                        ->options(function (callable $get) {
                            $user = Auth::user();
                            $query = \App\Models\Kamar::query();

                            // Filter by madrasah if selected
                            if ($madrasahId = $get('madrasah_id')) {
                                $query->where('madrasah_id', $madrasahId);
                            }

                            // Filter by wali kamar role
                            if ($user && $user->hasRole('wali kamar')) {
                                $query->where('wali_kamar_id', $user->id);
                            }

                            return $query->pluck('nama_kamar', 'id');
                        })
                        ->placeholder('Pilih Kamar (Opsional - Kosongkan untuk semua)')
                        ->searchable(),
                ])
                ->query(function (Builder $query, array $data): Builder {
                    return $query
                        ->when($data['madrasah_id'], fn ($q) => $q->whereHas('kamar', fn ($q) => $q->where('madrasah_id', $data['madrasah_id'])))
                        ->when($data['kamar_id'], fn ($q) => $q->where('kamar_id', $data['kamar_id']));
                }),
            Tables\Filters\Filter::make('jenis_presensi_asrama_id')
                ->form([
                    \Filament\Forms\Components\Select::make('jenis_presensi_asrama_id')
                        ->label('Jenis Presensi')
                        ->options(JenisPresensiAsrama::where('is_active', true)->pluck('nama', 'id')),
                ])
                ->query(function (Builder $query, array $data): Builder {
                    // Filter akan ditangani di getAttendanceCount method
                    return $query;
                }),
            Tables\Filters\Filter::make('tanggal')
                ->form([
                    \Filament\Forms\Components\DatePicker::make('dari')
                        ->label('Dari Tanggal')
                        ->default(now()->startOfMonth()),
                    \Filament\Forms\Components\DatePicker::make('sampai')
                        ->label('Sampai Tanggal')
                        ->default(now()),
                ]),
        ];
    }

    protected static function getAttendanceCount($siswa, $status, $table): int
    {
        $query = DetailPresensiAsrama::where('siswa_id', $siswa->id)
            ->where('status', $status);

        // Apply filters
        if ($tahunAjaranId = request()->input('tableFilters.tahun_ajaran_id.value')) {
            $query->whereHas('presensiAsrama', fn ($q) => $q->where('tahun_ajaran_id', $tahunAjaranId));
        }

        if ($jenisPresensiId = request()->input('tableFilters.jenis_presensi_asrama_id.value')) {
            $query->whereHas('presensiAsrama', fn ($q) => $q->where('jenis_presensi_asrama_id', $jenisPresensiId));
        }

        if ($dari = request()->input('tableFilters.tanggal.dari')) {
            $query->whereHas('presensiAsrama', fn ($q) => $q->where('tanggal', '>=', $dari));
        }

        if ($sampai = request()->input('tableFilters.tanggal.sampai')) {
            $query->whereHas('presensiAsrama', fn ($q) => $q->where('tanggal', '<=', $sampai));
        }

        return $query->count();
    }

    protected static function getTotalAttendance($siswa, $table): int
    {
        $query = DetailPresensiAsrama::where('siswa_id', $siswa->id);

        // Apply same filters
        if ($tahunAjaranId = request()->input('tableFilters.tahun_ajaran_id.value')) {
            $query->whereHas('presensiAsrama', fn ($q) => $q->where('tahun_ajaran_id', $tahunAjaranId));
        }

        if ($jenisPresensiId = request()->input('tableFilters.jenis_presensi_asrama_id.value')) {
            $query->whereHas('presensiAsrama', fn ($q) => $q->where('jenis_presensi_asrama_id', $jenisPresensiId));
        }

        if ($dari = request()->input('tableFilters.tanggal.dari')) {
            $query->whereHas('presensiAsrama', fn ($q) => $q->where('tanggal', '>=', $dari));
        }

        if ($sampai = request()->input('tableFilters.tanggal.sampai')) {
            $query->whereHas('presensiAsrama', fn ($q) => $q->where('tanggal', '<=', $sampai));
        }

        return $query->count();
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListLaporanPresensiAsramas::route('/'),
        ];
    }
}
