<?php

namespace App\Filament\Resources;

use App\Filament\Resources\KamarResource\Pages;
use App\Filament\Resources\KamarResource\RelationManagers;
use App\Models\Kamar;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Support\Facades\Auth;

class KamarResource extends Resource
{
    protected static ?string $model = Kamar::class;

    protected static ?string $navigationIcon = 'heroicon-o-cube';

    protected static ?string $pluralModelLabel = 'Kamar';

    protected static ?string $navigationGroup = 'Kegiatan Pengasuhan';

    protected static ?int $navigationSort = 39;

    public static function getEloquentQuery(): \Illuminate\Database\Eloquent\Builder
    {
        $query = parent::getEloquentQuery();
        $user = Auth::user();

        // Wali kamar hanya bisa lihat kamar yang dia pegang
        if ($user && $user->hasRole('wali kamar')) {
            $query->where('wali_kamar_id', $user->id);
        }

        return $query;
    }

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\Section::make('Detail Kamar')
                    ->description('Informasi dasar mengenai kamar.')
                    ->schema([
                        Forms\Components\Grid::make(2)
                            ->schema([
                                Forms\Components\Select::make('madrasah_id')
                                    ->relationship('madrasah', 'nama')
                                    ->required()
                                    ->label('Madrasah'),
                                Forms\Components\TextInput::make('nama_kamar')
                                    ->required()
                                    ->maxLength(50)
                                    ->label('Nama Kamar'),
                                Forms\Components\Select::make('jenis_kelamin')
                                    ->options([
                                        'L' => 'Laki-laki',
                                        'P' => 'Perempuan',
                                    ])
                                    ->required()
                                    ->label('Jenis Kelamin Penghuni'),
                                Forms\Components\TextInput::make('kapasitas')
                                    ->required()
                                    ->numeric()
                                    ->default(10)
                                    ->minValue(1)
                                    ->label('Kapasitas'),
                            ]),
                    ]),
                Forms\Components\Section::make('Penanggung Jawab')
                    ->description('Pilih wali kamar yang bertanggung jawab.')
                    ->schema([
                        Forms\Components\Select::make('wali_kamar_id')
                            ->relationship('waliKamar', 'name')
                            ->searchable()
                            ->preload()
                            ->default(null)
                            ->label('Wali Kamar'),
                    ]),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('madrasah.nama')
                    ->numeric()
                    ->sortable(),
                Tables\Columns\TextColumn::make('nama_kamar')
                    ->searchable(),
                Tables\Columns\TextColumn::make('jenis_kelamin')
                    ->label('Jenis Penghuni Kamar'),
                Tables\Columns\TextColumn::make('waliKamar.name')
                    ->label('Wali Kamar')
                    ->sortable()
                    ->searchable(),
                Tables\Columns\TextColumn::make('created_at')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
                Tables\Columns\TextColumn::make('updated_at')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
                Tables\Columns\TextColumn::make('deleted_at')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                //
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
                Tables\Actions\DeleteAction::make(),

            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            RelationManagers\SiswasRelationManager::class,
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListKamars::route('/'),
            'create' => Pages\CreateKamar::route('/create'),
            'edit' => Pages\EditKamar::route('/{record}/edit'),
        ];
    }
}
