<?php

namespace App\Filament\Pages;

use App\Models\DetailPresensiMadrasah;
use App\Models\JenisPresensiMadrasah;
use App\Models\Kelas;
use App\Models\PresensiMadrasah;
use App\Models\Siswa;
use App\Models\TahunAjaran;
use BezhanSalleh\FilamentShield\Traits\HasWidgetShield;
use Filament\Actions\Action;
use Filament\Forms;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Forms\Contracts\HasForms;
use Filament\Forms\Form;
use Filament\Notifications\Notification;
use Filament\Pages\Page;
use Illuminate\Support\Facades\Auth;
use Livewire\Attributes\Url;

class InputPresensiMadrasah extends Page implements HasForms
{
    use HasWidgetShield, InteractsWithForms;

    protected static ?string $navigationIcon = 'heroicon-o-document-text';

    protected static string $view = 'filament.pages.input-presensi-madrasah';

    protected static bool $shouldRegisterNavigation = false;

    public static function canAccess(): bool
    {
        return Auth::user()->can('page_InputPresensiMadrasah');
    }

    #[Url]
    public ?string $kelas_id = null;

    #[Url]
    public ?string $jenis_presensi_id = null;

    #[Url]
    public ?string $tanggal = null;

    public ?array $data = [];

    public function mount(): void
    {
        $user = Auth::user();

        // Auto-detect for Wali Kelas if parameters missing
        if (! $this->kelas_id && $user->hasRole('walikelas')) {
            $kelas = Kelas::where('wali_kelas_id', $user->id)->first();
            if ($kelas) {
                $this->kelas_id = $kelas->id;
            }
        }

        // Set default date if missing
        if (! $this->tanggal) {
            $this->tanggal = now()->toDateString();
        }

        $this->form->fill([
            'kelas_id' => $this->kelas_id,
            'jenis_presensi_id' => $this->jenis_presensi_id,
            'tanggal' => $this->tanggal,
        ]);

        if ($this->kelas_id && $this->jenis_presensi_id) {
            $this->loadData();
        }
    }

    public function loadData(): void
    {
        // Check if presensi already exists
        $presensi = PresensiMadrasah::where('kelas_id', $this->kelas_id)
            ->where('jenis_presensi_madrasah_id', $this->jenis_presensi_id)
            ->where('tanggal', $this->tanggal ?? now()->toDateString())
            ->first();

        if ($presensi) {
            // Load existing data
            $details = $presensi->detailPresensiMadrasahs;

            $this->data['students'] = $details->mapWithKeys(function ($item) {
                return [$item->siswa_id => $item->status];
            })->toArray();

            $this->data['keterangan'] = $details->mapWithKeys(function ($item) {
                return [$item->siswa_id => $item->keterangan];
            })->toArray();
        } else {
            // Initialize new data (default 'hadir')
            $siswas = Siswa::where('kelas_id', $this->kelas_id)->get();
            $this->data['students'] = $siswas->mapWithKeys(function ($item) {
                return [$item->id => 'hadir'];
            })->toArray();
        }
    }

    public function updated($property)
    {
        // When filter changes, reload data
        if (in_array($property, ['data.kelas_id', 'data.jenis_presensi_id', 'data.tanggal'])) {
            $this->kelas_id = $this->data['kelas_id'] ?? null;
            $this->jenis_presensi_id = $this->data['jenis_presensi_id'] ?? null;
            $this->tanggal = $this->data['tanggal'] ?? null;
            
            // Sync URL params
            $this->js("const url = new URL(window.location); url.searchParams.set('kelas_id', '{$this->kelas_id}'); url.searchParams.set('jenis_presensi_id', '{$this->jenis_presensi_id}'); url.searchParams.set('tanggal', '{$this->tanggal}'); window.history.pushState({}, '', url);");

            if ($this->kelas_id && $this->jenis_presensi_id) {
                $this->loadData();
            }
        }
    }

    public function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\Section::make('Filter Presensi')
                    ->schema([
                        Forms\Components\Select::make('kelas_id')
                            ->label('Kelas')
                            ->options(function () {
                                $user = Auth::user();
                                if ($user->hasRole('walikelas')) {
                                    return Kelas::where('wali_kelas_id', $user->id)->pluck('nama', 'id');
                                }
                                return Kelas::pluck('nama', 'id');
                            })
                            ->searchable()
                            ->preload()
                            ->required()
                            ->live()
                            ->afterStateUpdated(fn () => $this->updated('data.kelas_id')),
                        Forms\Components\Select::make('jenis_presensi_id')
                            ->label('Jenis Presensi')
                            ->options(JenisPresensiMadrasah::where('is_active', true)->pluck('nama', 'id'))
                            ->searchable()
                            ->preload()
                            ->required()
                            ->live()
                            ->afterStateUpdated(fn () => $this->updated('data.jenis_presensi_id')),
                        Forms\Components\DatePicker::make('tanggal')
                            ->label('Tanggal')
                            ->required()
                            ->live()
                            ->afterStateUpdated(fn () => $this->updated('data.tanggal')),
                    ])
                    ->columns(3),

                Forms\Components\Section::make('Daftar Siswa')
                    ->description('Pilih status kehadiran untuk setiap siswa. Keterangan akan muncul otomatis jika status bukan Hadir.')
                    ->schema(function () {
                        if (! $this->kelas_id) {
                            return [
                                Forms\Components\Placeholder::make('info')
                                    ->label('Info')
                                    ->content('Silakan pilih Kelas dan Jenis Presensi terlebih dahulu.'),
                            ];
                        }

                        $siswas = Siswa::where('kelas_id', $this->kelas_id)->get();
                        
                        if ($siswas->isEmpty()) {
                             return [
                                Forms\Components\Placeholder::make('empty')
                                    ->label('Info')
                                    ->content('Tidak ada siswa di kelas ini.'),
                            ];
                        }

                        // Split students into 2 groups
                        $total = $siswas->count();
                        $half = ceil($total / 2);
                        $leftGroup = $siswas->take($half);
                        $rightGroup = $siswas->skip($half);

                        $columns = [];

                        // Left Column
                        $leftFields = [];
                        foreach ($leftGroup as $siswa) {
                            $leftFields[] = Forms\Components\Fieldset::make($siswa->nama)
                                ->schema([
                                    Forms\Components\Select::make('students.'.$siswa->id)
                                        ->label('Status Kehadiran')
                                        ->options([
                                            'hadir' => 'Hadir',
                                            'sakit' => 'Sakit',
                                            'izin' => 'Izin',
                                            'terlambat' => 'Terlambat',
                                            'cabut' => 'Cabut',
                                            'alpha' => 'Alpha',
                                        ])
                                        ->default('hadir')
                                        ->required()
                                        ->live()
                                        ->native(false)
                                        ->columnSpanFull(),
                                    Forms\Components\Textarea::make('keterangan.'.$siswa->id)
                                        ->label('Keterangan')
                                        ->placeholder('Masukkan keterangan (opsional)')
                                        ->rows(2)
                                        ->columnSpanFull()
                                        ->visible(fn (Forms\Get $get) => $get('students.'.$siswa->id) !== 'hadir'
                                        ),
                                ])
                                ->columns(1);
                        }

                        // Right Column
                        $rightFields = [];
                        foreach ($rightGroup as $siswa) {
                            $rightFields[] = Forms\Components\Fieldset::make($siswa->nama)
                                ->schema([
                                    Forms\Components\Select::make('students.'.$siswa->id)
                                        ->label('Status Kehadiran')
                                        ->options([
                                            'hadir' => 'Hadir',
                                            'sakit' => 'Sakit',
                                            'izin' => 'Izin',
                                            'alpha' => 'Alpha',
                                        ])
                                        ->default('hadir')
                                        ->required()
                                        ->live()
                                        ->native(false)
                                        ->columnSpanFull(),
                                    Forms\Components\Textarea::make('keterangan.'.$siswa->id)
                                        ->label('Keterangan')
                                        ->placeholder('Masukkan keterangan (opsional)')
                                        ->rows(2)
                                        ->columnSpanFull()
                                        ->visible(fn (Forms\Get $get) => $get('students.'.$siswa->id) !== 'hadir'
                                        ),
                                ])
                                ->columns(1);
                        }

                        return [
                            Forms\Components\Grid::make(2)
                                ->schema([
                                    Forms\Components\Section::make('')
                                        ->schema($leftFields)
                                        ->columnSpan(1),
                                    Forms\Components\Section::make('')
                                        ->schema($rightFields)
                                        ->columnSpan(1),
                                ]),
                        ];
                    })
                    ->visible(fn () => $this->kelas_id && $this->jenis_presensi_id),
            ])
            ->statePath('data');
    }

    public function save(): void
    {
        $tahunAjaran = TahunAjaran::where('is_aktif', true)->first();

        if (! $tahunAjaran) {
            Notification::make()->title('Tahun Ajaran Aktif tidak ditemukan')->danger()->send();

            return;
        }

        // Find or Create Presensi Header
        $presensi = PresensiMadrasah::firstOrCreate(
            [
                'kelas_id' => $this->kelas_id,
                'jenis_presensi_madrasah_id' => $this->jenis_presensi_id,
                'tanggal' => $this->tanggal ?? now()->toDateString(),
            ],
            [
                'tahun_ajaran_id' => $tahunAjaran->id,
                'user_id' => Auth::id(),
                'is_locked' => false,
            ]
        );

        // Save Details
        foreach ($this->data['students'] as $siswaId => $status) {
            $keterangan = $this->data['keterangan'][$siswaId] ?? null;

            DetailPresensiMadrasah::updateOrCreate(
                [
                    'presensi_madrasah_id' => $presensi->id,
                    'siswa_id' => $siswaId,
                ],
                [
                    'status' => $status,
                    'keterangan' => $keterangan,
                ]
            );
        }

        Notification::make()
            ->title('Presensi berhasil disimpan')
            ->success()
            ->send();

        $this->redirectRoute('filament.admin.pages.dashboard');
    }

    public function getHeaderActions(): array
    {
        return [
            Action::make('save')
                ->label('Simpan Presensi')
                ->action('save')
                ->color('primary')
                ->visible(fn () => $this->kelas_id && $this->jenis_presensi_id),
        ];
    }

    public function getTitle(): string
    {
        if ($this->kelas_id && $this->jenis_presensi_id) {
            $kelas = Kelas::find($this->kelas_id);
            $jenis = JenisPresensiMadrasah::find($this->jenis_presensi_id);

            if ($kelas && $jenis) {
                return "Input Presensi: {$jenis->nama} - {$kelas->nama}";
            }
        }

        return 'Input Presensi Madrasah';
    }
}
