<?php

namespace App\Filament\Imports;

use App\Models\Kamar;
use App\Models\Kelas;
use App\Models\Madrasah;
use App\Models\Siswa;
use Filament\Actions\Imports\ImportColumn;
use Filament\Actions\Imports\Importer;
use Filament\Actions\Imports\Models\Import;

class SiswaImporter extends Importer
{
    protected static ?string $model = Siswa::class;

    public static function getColumns(): array
    {
        return [
            // Basic Info - Same order as export
            ImportColumn::make('nisn')
                ->label('NISN')
                ->requiredMapping()
                ->rules(['required', 'max:20'])
                ->example('1234567890'),

            ImportColumn::make('nis')
                ->label('NIS')
                ->rules(['nullable', 'max:20'])
                ->example('9876543210'),

            ImportColumn::make('nik')
                ->label('NIK')
                ->rules(['nullable', 'max:20'])
                ->example('3201012010010001'),

            ImportColumn::make('nama')
                ->label('Nama Lengkap')
                ->requiredMapping()
                ->rules(['required', 'max:255'])
                ->example('Ahmad Fauzi'),

            ImportColumn::make('jenis_kelamin')
                ->label('Jenis Kelamin')
                ->requiredMapping()
                ->rules(['required'])
                ->castStateUsing(function (string $state): string {
                    // Convert "Laki-laki" to "L" and "Perempuan" to "P"
                    $state = trim($state);
                    if (in_array($state, ['Laki-laki', 'laki-laki', 'LAKI-LAKI'])) {
                        return 'L';
                    }
                    if (in_array($state, ['Perempuan', 'perempuan', 'PEREMPUAN'])) {
                        return 'P';
                    }

                    // If already L or P, keep it
                    return strtoupper($state);
                })
                ->example('Laki-laki'),

            ImportColumn::make('tempat_lahir')
                ->label('Tempat Lahir')
                ->rules(['nullable', 'max:255'])
                ->example('Jakarta'),

            ImportColumn::make('tanggal_lahir')
                ->label('Tanggal Lahir')
                ->rules(['nullable'])
                ->castStateUsing(function (?string $state): ?string {
                    if (empty($state)) {
                        return null;
                    }
                    // Convert dd/mm/yyyy to Y-m-d for database
                    try {
                        $date = \Carbon\Carbon::createFromFormat('d/m/Y', $state);

                        return $date->format('Y-m-d');
                    } catch (\Exception $e) {
                        // Try other formats
                        try {
                            $date = \Carbon\Carbon::parse($state);

                            return $date->format('Y-m-d');
                        } catch (\Exception $e2) {
                            return null;
                        }
                    }
                })
                ->example('15/01/2010'),

            // Academic Info - Using database field names but accepting relation names from CSV
            ImportColumn::make('madrasah_id')
                ->label('Madrasah')
                ->requiredMapping()
                ->guess(['Madrasah', 'madrasah', 'MADRASAH', 'madrasah.nama', 'Nama Madrasah'])
                ->rules(['required'])
                ->castStateUsing(function (string $state): ?int {
                    // If it's already an ID, return it
                    if (is_numeric($state)) {
                        return (int) $state;
                    }
                    // Otherwise, lookup by name
                    $madrasah = Madrasah::where('nama', $state)->first();
                    if (! $madrasah) {
                        throw new \Exception("Madrasah '{$state}' tidak ditemukan");
                    }

                    return $madrasah->id;
                })
                ->example('MTs Kauman Muhammadiyah'),

            ImportColumn::make('kelas_id')
                ->label('Kelas')
                ->requiredMapping()
                ->guess(['Kelas', 'kelas', 'KELAS', 'kelas.nama', 'Nama Kelas'])
                ->rules(['required'])
                ->castStateUsing(function (string $state): ?int {
                    // If it's already an ID, return it
                    if (is_numeric($state)) {
                        return (int) $state;
                    }
                    // Otherwise, lookup by name
                    $kelas = Kelas::where('nama', $state)->first();
                    if (! $kelas) {
                        throw new \Exception("Kelas '{$state}' tidak ditemukan");
                    }

                    return $kelas->id;
                })
                ->example('VII A'),

            ImportColumn::make('status_tempat_tinggal')
                ->label('Status Tempat Tinggal')
                ->requiredMapping()
                ->rules(['required', 'max:255'])
                ->example('Asrama'),

            ImportColumn::make('kamar_id')
                ->label('Kamar')
                ->guess(['Kamar', 'kamar', 'KAMAR', 'kamar.nama_kamar', 'Nama Kamar'])
                ->rules(['nullable'])
                ->castStateUsing(function (?string $state): ?int {
                    if (empty($state) || trim($state) === '') {
                        return null;
                    }

                    $state = trim($state);

                    // If it's already an ID, return it
                    if (is_numeric($state)) {
                        return (int) $state;
                    }

                    // Otherwise, lookup by name
                    $kamar = Kamar::where('nama_kamar', $state)->first();

                    // If kamar not found, log warning but don't fail (since it's nullable)
                    if (! $kamar) {
                        \Log::warning("Kamar '{$state}' tidak ditemukan, kamar_id akan di-set null");

                        return null;
                    }

                    return $kamar->id;
                })
                ->example('Kamar 101'),

            ImportColumn::make('no_kk')
                ->label('No. KK')
                ->rules(['nullable', 'max:20'])
                ->example('3201011234567890'),

            // Father Info
            ImportColumn::make('nama_ayah')
                ->label('Nama Ayah')
                ->rules(['nullable', 'max:255'])
                ->example('Budi Santoso'),

            ImportColumn::make('nik_ayah')
                ->label('NIK Ayah')
                ->rules(['nullable', 'max:20'])
                ->example('3201011980010001'),

            ImportColumn::make('pendidikan_ayah')
                ->label('Pendidikan Ayah')
                ->rules(['nullable', 'max:255'])
                ->example('S1'),

            ImportColumn::make('pekerjaan_ayah')
                ->label('Pekerjaan Ayah')
                ->rules(['nullable', 'max:255'])
                ->example('Wiraswasta'),

            // Mother Info
            ImportColumn::make('nama_ibu')
                ->label('Nama Ibu')
                ->rules(['nullable', 'max:255'])
                ->example('Siti Aminah'),

            ImportColumn::make('nik_ibu')
                ->label('NIK Ibu')
                ->rules(['nullable', 'max:20'])
                ->example('3201011985010001'),

            ImportColumn::make('pendidikan_ibu')
                ->label('Pendidikan Ibu')
                ->rules(['nullable', 'max:255'])
                ->example('SMA'),

            ImportColumn::make('pekerjaan_ibu')
                ->label('Pekerjaan Ibu')
                ->rules(['nullable', 'max:255'])
                ->example('Ibu Rumah Tangga'),

            ImportColumn::make('no_hp_ortu')
                ->label('No. HP Orang Tua')
                ->rules(['nullable', 'max:20'])
                ->example('081234567890'),

            // Address Info
            ImportColumn::make('alamat')
                ->label('Alamat')
                ->rules(['nullable'])
                ->example('Jl. Merdeka No. 123'),

            ImportColumn::make('provinsi')
                ->label('Provinsi')
                ->rules(['nullable', 'max:255'])
                ->example('Jawa Barat'),

            ImportColumn::make('kabupaten_kota')
                ->label('Kabupaten/Kota')
                ->rules(['nullable', 'max:255'])
                ->example('Bandung'),

            ImportColumn::make('kecamatan')
                ->label('Kecamatan')
                ->rules(['nullable', 'max:255'])
                ->example('Coblong'),

            ImportColumn::make('desa_kelurahan')
                ->label('Desa/Kelurahan')
                ->rules(['nullable', 'max:255'])
                ->example('Dago'),

            ImportColumn::make('kode_pos')
                ->label('Kode Pos')
                ->rules(['nullable', 'max:10'])
                ->example('40135'),
        ];
    }

    public function resolveRecord(): ?Siswa
    {
        // Upsert based on NISN as unique identifier
        return Siswa::firstOrNew([
            'nisn' => $this->data['nisn'],
        ]);
    }

    public static function getCompletedNotificationBody(Import $import): string
    {
        $body = 'Import data siswa selesai dan '.number_format($import->successful_rows).' '.str('baris')->plural($import->successful_rows).' berhasil diimpor.';

        if ($failedRowsCount = $import->getFailedRowsCount()) {
            $body .= ' '.number_format($failedRowsCount).' '.str('baris')->plural($failedRowsCount).' gagal diimpor.';
        }

        return $body;
    }
}
