<?php

namespace App\Filament\Exports;

use App\Models\User;
use Filament\Actions\Exports\ExportColumn;
use Filament\Actions\Exports\Exporter;
use Filament\Actions\Exports\Models\Export;

class UserExporter extends Exporter
{
    protected static ?string $model = User::class;

    public function getQuery(): \Illuminate\Database\Eloquent\Builder
    {
        return User::query()
            ->with(['madrasah', 'roles']) // 👈 EAGER LOAD relasi
            ->orderBy('created_at', 'desc');
    }

    public static function getColumns(): array
    {
        return [
            ExportColumn::make('id')
                ->label('ID'),

            ExportColumn::make('name')
                ->label('Nama Lengkap'),

            ExportColumn::make('email')
                ->label('Email'),

            // ✅ KOLOM BARU: Madrasah dengan handling nullable
            ExportColumn::make('madrasah.nama')
                ->label('Nama Madrasah')
                ->default('Tidak Ada Madrasah'), // 👈 Handle null gracefully

            ExportColumn::make('madrasah_id')
                ->label('ID Madrasah')
                ->default('-'),

            // ✅ KOLOM BARU: Roles (multiple roles)
            ExportColumn::make('roles')
                ->label('Peran')
                ->formatStateUsing(function ($state, User $record) {
                    // $state adalah collection roles
                    // Ambil semua nama role, join dengan koma
                    return $record->roles->pluck('name')->join(', ') ?: 'Tidak Ada Role';
                }),

            // ✅ KOLOM YANG DIAKTIFKAN: Timestamps
            ExportColumn::make('email_verified_at')
                ->label('Email Terverifikasi')
                ->formatStateUsing(fn ($state) => $state ? $state->format('d/m/Y H:i') : 'Belum Verifikasi'),

            ExportColumn::make('created_at')
                ->label('Tanggal Dibuat')
                ->formatStateUsing(fn ($state) => $state->format('d/m/Y H:i')),

            ExportColumn::make('updated_at')
                ->label('Terakhir Diupdate')
                ->formatStateUsing(fn ($state) => $state->format('d/m/Y H:i')),

            // Avatar (optional, bisa dihapus jika tidak perlu)
            ExportColumn::make('avatar_url')
                ->label('Avatar URL')
                ->default('-'),
        ];
    }

    public static function getCompletedNotificationBody(Export $export): string
    {
        $successCount = number_format($export->successful_rows);
        $successPlural = str('baris')->plural($export->successful_rows);

        $body = "Export data user telah selesai! {$successCount} {$successPlural} berhasil diekspor.";

        if ($failedRowsCount = $export->getFailedRowsCount()) {
            $failedCount = number_format($failedRowsCount);
            $failedPlural = str('baris')->plural($failedRowsCount);
            $body .= " {$failedCount} {$failedPlural} gagal diekspor.";
        }

        return $body;
    }

    public function getFailedNotificationBody(Export $export): string
    {
        return 'Export data user gagal! Silakan coba lagi atau hubungi administrator.';
    }

    /**
     * ✅ OPSIONAL: Limit chunk size untuk performa
     * Default 100, bisa disesuaikan
     */
    public function getChunkSize(): int
    {
        return 500; // Export 500 records per batch
    }

    /**
     * ✅ OPSIONAL: Custom filename
     */
    public function getFileName(Export $export): string
    {
        return 'users-export-'.now()->format('Y-m-d-His').'.xlsx';
    }
}
