<?php

namespace App\Exports;

use App\Models\Kamar;
use App\Models\Kelas;
use Illuminate\Support\Facades\Auth;
use Maatwebsite\Excel\Concerns\FromArray;
use Maatwebsite\Excel\Concerns\ShouldAutoSize;
use Maatwebsite\Excel\Concerns\WithColumnFormatting;
use Maatwebsite\Excel\Concerns\WithColumnWidths;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithStyles;
use PhpOffice\PhpSpreadsheet\Style\Alignment;
use PhpOffice\PhpSpreadsheet\Style\Fill;
use PhpOffice\PhpSpreadsheet\Style\NumberFormat;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;

class SiswaTemplateExport implements FromArray, ShouldAutoSize, WithColumnFormatting, WithColumnWidths, WithHeadings, WithStyles
{
    protected $madrasahId;

    public function __construct($madrasahId = null)
    {
        $this->madrasahId = $madrasahId ?? Auth::user()->madrasah_id;
    }

    public function array(): array
    {
        // Ambil contoh data kelas dan kamar untuk referensi
        $kelasContoh = Kelas::where('madrasah_id', $this->madrasahId)->first();
        $kamarContoh = Kamar::first();

        // Berikan 2 baris contoh dengan NISN berbeda
        return [
            [
                1,  // madrasah_id (1=MTs, 2=MA)
                "'1234567890",  // nisn - prefix ' agar Excel treat as text
                "'001",  // nis
                "'3201234567890123",  // nik
                'Ahmad Fauzi',  // nama
                'L',  // jenis_kelamin
                'Jakarta',  // tempat_lahir
                '2010-01-15',  // tanggal_lahir
                $kelasContoh ? $kelasContoh->nama : 'VII T PA',  // nama_kelas
                'asrama',  // status_tempat_tinggal
                $kamarContoh ? $kamarContoh->nama_kamar : 'Kamar 1',  // nama_kamar
                "'3201234567890000",  // no_kk
                'Budi Santoso',  // nama_ayah
                "'3201234567890001",  // nik_ayah
                'S1',  // pendidikan_ayah
                'PNS',  // pekerjaan_ayah
                'Siti Aminah',  // nama_ibu
                "'3201234567890002",  // nik_ibu
                'SMA',  // pendidikan_ibu
                'Ibu Rumah Tangga',  // pekerjaan_ibu
                "'081234567890",  // no_hp_ortu
                'Jl. Merdeka No. 123',  // alamat
                'DKI Jakarta',  // provinsi
                'Jakarta Pusat',  // kabupaten_kota
                'Menteng',  // kecamatan
                'Menteng',  // desa_kelurahan
                "'10310",  // kode_pos
            ],
            [
                2,  // madrasah_id (1=MTs, 2=MA)
                "'0987654321",  // NISN berbeda agar tidak duplikat
                "'002",
                "'3301234567890123",
                'Siti Nurhaliza',
                'P',
                'Bandung',
                '2010-05-20',
                $kelasContoh ? $kelasContoh->nama : 'X ITT',
                'non_asrama',
                '',  // kamar kosong untuk non asrama
                "'3301234567890000",
                'Joko Widodo',
                "'3301234567890001",
                'SMA',
                'Wiraswasta',
                'Mega Wati',
                "'3301234567890002",
                'S1',
                'Guru',
                "'082345678901",
                'Jl. Asia Afrika No. 45',
                'Jawa Barat',
                'Bandung',
                'Coblong',
                'Dago',
                "'40135",
            ],
        ];
    }

    public function headings(): array
    {
        return [
            'madrasah_id',
            'nisn',
            'nis',
            'nik',
            'nama',
            'jenis_kelamin',
            'tempat_lahir',
            'tanggal_lahir',
            'nama_kelas',
            'status_tempat_tinggal',
            'nama_kamar',
            'no_kk',
            'nama_ayah',
            'nik_ayah',
            'pendidikan_ayah',
            'pekerjaan_ayah',
            'nama_ibu',
            'nik_ibu',
            'pendidikan_ibu',
            'pekerjaan_ibu',
            'no_hp_ortu',
            'alamat',
            'provinsi',
            'kabupaten_kota',
            'kecamatan',
            'desa_kelurahan',
            'kode_pos',
        ];
    }

    public function styles(Worksheet $sheet)
    {
        return [
            1 => [
                'font' => [
                    'bold' => true,
                    'color' => ['rgb' => 'FFFFFF'],
                ],
                'fill' => [
                    'fillType' => Fill::FILL_SOLID,
                    'startColor' => ['rgb' => '4472C4'],
                ],
                'alignment' => [
                    'horizontal' => Alignment::HORIZONTAL_CENTER,
                    'vertical' => Alignment::VERTICAL_CENTER,
                ],
            ],
        ];
    }

    public function columnWidths(): array
    {
        return [
            'A' => 12,  // madrasah_id
            'B' => 15,  // NISN
            'C' => 10,  // NIS
            'D' => 18,  // NIK
            'E' => 25,  // Nama
            'F' => 15,  // Jenis Kelamin
            'G' => 15,  // Tempat Lahir
            'H' => 15,  // Tanggal Lahir
            'I' => 15,  // Nama Kelas
            'J' => 20,  // Status Tempat Tinggal
            'K' => 15,  // Nama Kamar
            'L' => 18,  // No KK
            'M' => 20,  // Nama Ayah
            'N' => 18,  // NIK Ayah
            'O' => 18,  // Pendidikan Ayah
            'P' => 18,  // Pekerjaan Ayah
            'Q' => 20,  // Nama Ibu
            'R' => 18,  // NIK Ibu
            'S' => 18,  // Pendidikan Ibu
            'T' => 18,  // Pekerjaan Ibu
            'U' => 30,  // Al HP Orang Tua
            'V' => 30,  // Alaminsi
            'W' => 15,  // Provinsi
            'X' => 18,  // Kabupaten/Kota
            'Y' => 15,  // Kecamatan
            'Z' => 18,  // Desa/Kelurahan
            'AA' => 12,  // Kode Pos
        ];
    }

    public function columnFormats(): array
    {
        return [
            'B' => NumberFormat::FORMAT_TEXT,  // NISN sebagai text
            'C' => NumberFormat::FORMAT_TEXT,  // NIS sebagai text
            'D' => NumberFormat::FORMAT_TEXT,  // NIK sebagai text
            'L' => NumberFormat::FORMAT_TEXT,  // No KK sebagai text
            'N' => NumberFormat::FORMAT_TEXT,  // NIK Ayah sebagai text
            'R' => NumberFormat::FORMAT_TEXT,  // NIK Ibu sebagai text
            'U' => NumberFormat::FORMAT_TEXT,  // No HP sebagai text
            'AA' => NumberFormat::FORMAT_TEXT,  // Kode Pos sebagai text
        ];
    }
}
