<?php

namespace App\Exports;

use App\Models\DetailPresensiMadrasah;
use App\Models\JenisPresensiMadrasah;
use App\Models\Kelas;
use Illuminate\Contracts\View\View;
use Illuminate\Support\Facades\Auth;
use Maatwebsite\Excel\Concerns\FromView;

class LaporanMadrasahExport implements FromView
{
    protected $filters;

    public function __construct(array $filters)
    {
        $this->filters = $filters;
    }

    public function view(): View
    {
        $data = $this->prepareData();

        return view('exports.laporan-madrasah-pdf', $data);
    }

    protected function prepareData(): array
    {
        $filters = $this->filters;
        $user = Auth::user();

        // 1. Fetch Classes (Kelas)
        $kelasQuery = Kelas::query()->with('waliKelas');

        // Filter by Madrasah if selected
        if (! empty($filters['madrasah_id'])) {
            $kelasQuery->where('madrasah_id', $filters['madrasah_id']);
        }

        // Filter by specific Class if selected
        if (! empty($filters['kelas_id'])) {
            $kelasQuery->where('id', $filters['kelas_id']);
        } else {
            // Role Scoping for Classes
            if ($user->hasRole('walikelas')) {
                $kelasQuery->where('wali_kelas_id', $user->id);
            } elseif (! $user->hasRole(['super_admin', 'pimpinan', 'operator'])) {
                $madrasahId = $user->getMadrasahId();
                if ($madrasahId) {
                    $kelasQuery->where('madrasah_id', $madrasahId);
                }
            }

            // Filter by Active Tab (MTs vs MA)
            // Note: This matches the widget logic we fixed earlier
            $activeTab = $filters['activeTab'] ?? null;
            // Fallback from Referer handled in controller/resource, but ensure it's passed here

            if ($activeTab === 'mts') {
                $kelasQuery->whereIn('tingkat', ['VII', 'VIII', 'IX']);
            } elseif ($activeTab === 'ma') {
                $kelasQuery->whereIn('tingkat', ['X', 'XI', 'XII']);
            }
        }

        $classes = $kelasQuery->orderBy('tingkat')->orderBy('nama')->get();

        // 2. Fetch Jenis Presensi
        $jenisQuery = JenisPresensiMadrasah::where('is_active', true);
        if (! empty($filters['jenis_presensi_madrasah_id'])) {
            $jenisQuery->where('id', $filters['jenis_presensi_madrasah_id']);
        }
        $jenisPresensis = $jenisQuery->get();

        // 3. Eager Load Students for each Class (Without Presensi Models)
        foreach ($classes as $kelas) {
            $kelas->setRelation('siswas',
                $kelas->siswas()->orderBy('nama')->get()
            );
        }

        // 4. Fetch Aggregates Efficiently
        // Instead of loading 100k+ models, we query the counts directly
        $siswaIds = $classes->pluck('siswas')->flatten()->pluck('id')->unique()->toArray();

        $aggregates = DetailPresensiMadrasah::query()
            ->selectRaw('siswa_id, presensi_madrasahs.jenis_presensi_madrasah_id as jenis_id, status, count(*) as total')
            ->join('presensi_madrasahs', 'detail_presensi_madrasahs.presensi_madrasah_id', '=', 'presensi_madrasahs.id')
            ->whereIn('siswa_id', $siswaIds)
            ->where(function ($q) use ($filters) {
                if (! empty($filters['tahun_ajaran_id'])) {
                    $q->where('presensi_madrasahs.tahun_ajaran_id', $filters['tahun_ajaran_id']);
                }
                if (! empty($filters['dari'])) {
                    $q->where('presensi_madrasahs.tanggal', '>=', $filters['dari']);
                }
                if (! empty($filters['sampai'])) {
                    $q->where('presensi_madrasahs.tanggal', '<=', $filters['sampai']);
                }
                // Filter by Jenis Presensi if specified
                if (! empty($filters['jenis_presensi_madrasah_id'])) {
                    $q->where('presensi_madrasahs.jenis_presensi_madrasah_id', $filters['jenis_presensi_madrasah_id']);
                }
            })
            ->groupBy('siswa_id', 'jenis_id', 'status')
            ->get();

        // Transform aggregates into a fast lookup array [siswa_id][jenis_id][status] = count
        $attendanceData = [];
        foreach ($aggregates as $row) {
            $attendanceData[$row->siswa_id][$row->jenis_id][$row->status] = $row->total;
        }

        // 5. Fetch Keterangan for non-hadir status
        $keteranganData = DetailPresensiMadrasah::query()
            ->selectRaw('detail_presensi_madrasahs.siswa_id, presensi_madrasahs.jenis_presensi_madrasah_id as jenis_id, detail_presensi_madrasahs.status, detail_presensi_madrasahs.keterangan, presensi_madrasahs.tanggal')
            ->join('presensi_madrasahs', 'detail_presensi_madrasahs.presensi_madrasah_id', '=', 'presensi_madrasahs.id')
            ->whereIn('detail_presensi_madrasahs.siswa_id', $siswaIds)
            ->whereIn('detail_presensi_madrasahs.status', ['sakit', 'izin', 'alpha', 'cabut', 'terlambat'])
            ->whereNotNull('detail_presensi_madrasahs.keterangan')
            ->where('detail_presensi_madrasahs.keterangan', '!=', '')
            ->where(function ($q) use ($filters) {
                if (! empty($filters['tahun_ajaran_id'])) {
                    $q->where('presensi_madrasahs.tahun_ajaran_id', $filters['tahun_ajaran_id']);
                }
                if (! empty($filters['dari'])) {
                    $q->where('presensi_madrasahs.tanggal', '>=', $filters['dari']);
                }
                if (! empty($filters['sampai'])) {
                    $q->where('presensi_madrasahs.tanggal', '<=', $filters['sampai']);
                }
                if (! empty($filters['jenis_presensi_madrasah_id'])) {
                    $q->where('presensi_madrasahs.jenis_presensi_madrasah_id', $filters['jenis_presensi_madrasah_id']);
                }
            })
            ->orderBy('presensi_madrasahs.tanggal', 'desc')
            ->get();

        // Group keterangan by siswa and jenis: [siswa_id][jenis_id] = array of keterangan
        $keteranganGrouped = [];
        foreach ($keteranganData as $row) {
            if (! isset($keteranganGrouped[$row->siswa_id][$row->jenis_id])) {
                $keteranganGrouped[$row->siswa_id][$row->jenis_id] = [];
            }
            $keteranganGrouped[$row->siswa_id][$row->jenis_id][] = [
                'status' => $row->status,
                'keterangan' => $row->keterangan,
                'tanggal' => $row->tanggal,
            ];
        }

        return [
            'classes' => $classes,
            'jenisPresensis' => $jenisPresensis,
            'attendanceData' => $attendanceData,
            'keteranganData' => $keteranganGrouped,
            'filters' => $filters,
            'user' => $user,
        ];
    }
}
