<?php

namespace App\Exports;

use App\Models\DetailPresensiAsrama;
use App\Models\JenisPresensiAsrama;
use App\Models\Kamar;
use Illuminate\Contracts\View\View;
use Illuminate\Support\Facades\Auth;
use Maatwebsite\Excel\Concerns\FromView;

class LaporanAsramaExport implements FromView
{
    protected $filters;

    protected $format;

    public function __construct(array $filters, string $format = 'pdf')
    {
        $this->filters = $filters;
        $this->format = $format;
    }

    public function view(): View
    {
        $data = $this->prepareData();

        // Use different view based on format
        $viewName = $this->format === 'excel'
            ? 'exports.laporan-asrama-excel'
            : 'exports.laporan-asrama-pdf';

        return view($viewName, $data);
    }

    protected function prepareData(): array
    {
        $filters = $this->filters;
        $user = Auth::user();

        // 1. Fetch Kamars with role-based filtering
        $kamarQuery = Kamar::query()->with(['waliKamar', 'madrasah']);

        // Filter by Madrasah if selected
        if (! empty($filters['madrasah_id'])) {
            $kamarQuery->where('madrasah_id', $filters['madrasah_id']);
        }

        // Filter by specific Kamar if selected
        if (! empty($filters['kamar_id'])) {
            $kamarQuery->where('id', $filters['kamar_id']);
        } else {
            // Role Scoping for Kamars
            if ($user->hasRole('wali kamar')) {
                // Wali kamar only sees their own kamar
                $kamarQuery->where('wali_kamar_id', $user->id);
            } elseif (! $user->hasRole(['super_admin', 'pimpinan', 'operator'])) {
                // Regular users see only their madrasah's kamars
                $madrasahId = $user->getMadrasahId();
                if ($madrasahId) {
                    $kamarQuery->where('madrasah_id', $madrasahId);
                }
            }
            // super_admin, pimpinan, operator see all kamars (no filtering)
        }

        // Eager load siswas in one query to avoid N+1
        $kamars = $kamarQuery->with(['siswas' => function ($q) {
            $q->orderBy('nama');
        }])->orderBy('nama_kamar')->get();

        // 2. Fetch Jenis Presensi Asrama
        $jenisQuery = JenisPresensiAsrama::where('is_active', true);
        if (! empty($filters['jenis_presensi_asrama_id'])) {
            $jenisQuery->where('id', $filters['jenis_presensi_asrama_id']);
        }
        $jenisPresensis = $jenisQuery->get();

        // 3. Fetch Aggregates Efficiently
        // Instead of loading 100k+ models, we query the counts directly
        $siswaIds = $kamars->pluck('siswas')->flatten()->pluck('id')->unique()->toArray();

        // Skip if no students found
        if (empty($siswaIds)) {
            return [
                'kamars' => $kamars,
                'jenisPresensis' => $jenisPresensis,
                'attendanceData' => [],
                'filters' => $filters,
                'user' => $user,
            ];
        }

        $aggregates = DetailPresensiAsrama::query()
            ->selectRaw('siswa_id, presensi_asramas.jenis_presensi_asrama_id as jenis_id, status, count(*) as total')
            ->join('presensi_asramas', 'detail_presensi_asramas.presensi_asrama_id', '=', 'presensi_asramas.id')
            ->whereIn('siswa_id', $siswaIds)
            ->where(function ($q) use ($filters) {
                if (! empty($filters['tahun_ajaran_id'])) {
                    $q->where('presensi_asramas.tahun_ajaran_id', $filters['tahun_ajaran_id']);
                }
                if (! empty($filters['dari'])) {
                    $q->where('presensi_asramas.tanggal', '>=', $filters['dari']);
                }
                if (! empty($filters['sampai'])) {
                    $q->where('presensi_asramas.tanggal', '<=', $filters['sampai']);
                }
                // Filter by Jenis Presensi if specified
                if (! empty($filters['jenis_presensi_asrama_id'])) {
                    $q->where('presensi_asramas.jenis_presensi_asrama_id', $filters['jenis_presensi_asrama_id']);
                }
            })
            ->groupBy('siswa_id', 'jenis_id', 'status')
            ->get();

        // Transform aggregates into a fast lookup array [siswa_id][jenis_id][status] = count
        $attendanceData = [];
        foreach ($aggregates as $row) {
            $attendanceData[$row->siswa_id][$row->jenis_id][$row->status] = $row->total;
        }

        // 4. Fetch Keterangan for non-hadir status
        $keteranganData = DetailPresensiAsrama::query()
            ->selectRaw('detail_presensi_asramas.siswa_id, presensi_asramas.jenis_presensi_asrama_id as jenis_id, detail_presensi_asramas.status, detail_presensi_asramas.keterangan, presensi_asramas.tanggal')
            ->join('presensi_asramas', 'detail_presensi_asramas.presensi_asrama_id', '=', 'presensi_asramas.id')
            ->whereIn('detail_presensi_asramas.siswa_id', $siswaIds)
            ->whereIn('detail_presensi_asramas.status', ['sakit', 'izin', 'alpha'])
            ->whereNotNull('detail_presensi_asramas.keterangan')
            ->where('detail_presensi_asramas.keterangan', '!=', '')
            ->where(function ($q) use ($filters) {
                if (! empty($filters['tahun_ajaran_id'])) {
                    $q->where('presensi_asramas.tahun_ajaran_id', $filters['tahun_ajaran_id']);
                }
                if (! empty($filters['dari'])) {
                    $q->where('presensi_asramas.tanggal', '>=', $filters['dari']);
                }
                if (! empty($filters['sampai'])) {
                    $q->where('presensi_asramas.tanggal', '<=', $filters['sampai']);
                }
                if (! empty($filters['jenis_presensi_asrama_id'])) {
                    $q->where('presensi_asramas.jenis_presensi_asrama_id', $filters['jenis_presensi_asrama_id']);
                }
            })
            ->orderBy('presensi_asramas.tanggal', 'desc')
            ->get();

        // Group keterangan by siswa and jenis: [siswa_id][jenis_id] = array of keterangan
        $keteranganGrouped = [];
        foreach ($keteranganData as $row) {
            if (!isset($keteranganGrouped[$row->siswa_id][$row->jenis_id])) {
                $keteranganGrouped[$row->siswa_id][$row->jenis_id] = [];
            }
            $keteranganGrouped[$row->siswa_id][$row->jenis_id][] = [
                'status' => $row->status,
                'keterangan' => $row->keterangan,
                'tanggal' => $row->tanggal,
            ];
        }

        return [
            'kamars' => $kamars,
            'jenisPresensis' => $jenisPresensis,
            'attendanceData' => $attendanceData,
            'keteranganData' => $keteranganGrouped,
            'filters' => $filters,
            'user' => $user,
        ];
    }
}
