# Design Document - Siswa Export & Import Feature

## Overview

Fitur ini menambahkan kemampuan export dan import data siswa di Filament v3 Resource. Export menggunakan sistem export bawaan Filament dengan Exporter class, sedangkan import menggunakan sistem import bawaan Filament dengan Importer class. Kedua fitur ini akan terintegrasi dengan SiswaResource yang sudah ada dan memproses data di background menggunakan queue untuk performa optimal.

## Architecture

### High-Level Architecture

```
┌─────────────────────────────────────────────────────────────┐
│                     Filament Admin Panel                     │
├─────────────────────────────────────────────────────────────┤
│                                                               │
│  ┌──────────────────────────────────────────────────────┐  │
│  │           SiswaResource (List Page)                   │  │
│  │                                                        │  │
│  │  ┌──────────────┐  ┌──────────────┐                 │  │
│  │  │ Export Action│  │ Import Action│                 │  │
│  │  └──────┬───────┘  └──────┬───────┘                 │  │
│  └─────────┼──────────────────┼──────────────────────────┘  │
│            │                  │                              │
└────────────┼──────────────────┼──────────────────────────────┘
             │                  │
             ▼                  ▼
    ┌────────────────┐  ┌────────────────┐
    │ SiswaExporter  │  │ SiswaImporter  │
    └────────┬───────┘  └────────┬───────┘
             │                   │
             ▼                   ▼
    ┌─────────────────────────────────────┐
    │         Queue System                 │
    │  (Export/Import Background Jobs)     │
    └─────────┬───────────────────────────┘
              │
              ▼
    ┌─────────────────────────────────────┐
    │      Storage (Export Files)          │
    │      Database (Siswa Records)        │
    └─────────────────────────────────────┘
```

### Component Interaction Flow

**Export Flow:**
1. User clicks "Export" button in SiswaResource list page
2. Filament displays export dialog with column selection
3. User selects columns and confirms export
4. ExportAction dispatches export job to queue
5. SiswaExporter processes data in background
6. Export file is saved to storage
7. User receives notification with download link
8. User downloads file from storage

**Import Flow:**
1. User clicks "Import" button in SiswaResource list page
2. Filament displays import dialog with file upload
3. User uploads CSV file
4. ImportAction validates file format
5. ImportAction dispatches import job to queue
6. SiswaImporter processes CSV rows in background
7. Valid rows are inserted/updated in database
8. Invalid rows are logged as failed
9. User receives notification with import summary
10. User can download failed rows report if any

## Components and Interfaces

### 1. SiswaExporter

**Location:** `app/Filament/Exports/SiswaExporter.php`

**Purpose:** Defines columns to be exported and handles data transformation for export.

**Key Methods:**
- `getColumns()`: Returns array of ExportColumn definitions
- `getCompletedNotificationBody()`: Returns notification message when export completes

**Columns to Export:**
- Basic Info: nisn, nis, nik, nama, jenis_kelamin, tempat_lahir, tanggal_lahir
- Academic: madrasah.nama, kelas.nama, status_tempat_tinggal, kamar.nama_kamar
- Family: nama_ayah, nik_ayah, pendidikan_ayah, pekerjaan_ayah, nama_ibu, nik_ibu, pendidikan_ibu, pekerjaan_ibu, no_hp_ortu, no_kk
- Address: alamat, provinsi, kabupaten_kota, kecamatan, desa_kelurahan, kode_pos

### 2. SiswaImporter

**Location:** `app/Filament/Imports/SiswaImporter.php`

**Purpose:** Defines columns to be imported and handles data validation and transformation for import.

**Key Methods:**
- `getColumns()`: Returns array of ImportColumn definitions with validation rules
- `resolveRecord()`: Handles record creation/update logic
- `getCompletedNotificationBody()`: Returns notification message when import completes

**Validation Rules:**
- Required fields: nama, nisn, madrasah_id, kelas_id, jenis_kelamin, status_tempat_tinggal
- jenis_kelamin: must be 'L' or 'P'
- tanggal_lahir: must be valid date format
- madrasah_id, kelas_id, kamar_id: must exist in database
- nisn: unique identifier for upsert logic

### 3. SiswaResource Modifications

**Location:** `app/Filament/Resources/SiswaResource.php`

**Modifications:**
- Add `ExportAction` to header actions
- Add `ImportAction` to header actions
- Configure actions to use SiswaExporter and SiswaImporter

### 4. Export Template Generator

**Purpose:** Generate CSV template for import with correct headers and sample data.

**Implementation:** Static method in SiswaImporter or separate helper class.

## Data Models

### Siswa Model (Existing)

**Table:** `siswas`

**Key Fields:**
```php
- id: bigint (primary key)
- madrasah_id: bigint (foreign key, required)
- kelas_id: bigint (foreign key, required)
- kamar_id: bigint (foreign key, nullable)
- nisn: string(20) (required, unique)
- nis: string(20) (nullable)
- nik: string(20) (nullable)
- nama: string(255) (required)
- jenis_kelamin: enum('L','P') (required)
- tempat_lahir: string(255) (nullable)
- tanggal_lahir: date (nullable)
- status_tempat_tinggal: enum (required)
- no_kk: string(20) (nullable)
- nama_ayah: string(255) (nullable)
- nik_ayah: string(20) (nullable)
- pendidikan_ayah: string(255) (nullable)
- pekerjaan_ayah: string(255) (nullable)
- nama_ibu: string(255) (nullable)
- nik_ibu: string(20) (nullable)
- pendidikan_ibu: string(255) (nullable)
- pekerjaan_ibu: string(255) (nullable)
- no_hp_ortu: string(20) (nullable)
- alamat: text (nullable)
- provinsi: string(255) (nullable)
- kabupaten_kota: string(255) (nullable)
- kecamatan: string(255) (nullable)
- desa_kelurahan: string(255) (nullable)
- kode_pos: string(10) (nullable)

```

**Relationships:**
- belongsTo: Madrasah
- belongsTo: Kelas
- belongsTo: Kamar

### Export Model (Filament Built-in)

**Table:** `exports`

**Purpose:** Tracks export jobs and stores metadata.

**Key Fields:**
- id, user_id, exporter, total_rows, successful_rows, file_disk, file_name, completed_at, etc.

### FailedImportRow Model (Filament Built-in)

**Table:** `failed_import_rows`

**Purpose:** Stores failed import rows with error messages.

**Key Fields:**
- id, import_id, data, validation_error, etc.

## Correctness Properties

*A property is a characteristic or behavior that should hold true across all valid executions of a system-essentially, a formal statement about what the system should do. Properties serve as the bridge between human-readable specifications and machine-verifiable correctness guarantees.*


### Export Properties

Property 1: Export job queuing
*For any* export action initiated, a background job should be created in the queue system
**Validates: Requirements 1.3**

Property 2: Export completion notification
*For any* completed export, a notification with download link should be sent to the user
**Validates: Requirements 1.4**

Property 3: Column preference persistence
*For any* saved column selection, the same selection should be used in subsequent exports
**Validates: Requirements 1.2, 2.4**

Property 4: Filtered export accuracy
*For any* active table filter, the exported data should contain only records matching that filter
**Validates: Requirements 3.2**

Property 5: Export status transitions
*For any* export job, the status should correctly transition from "processing" to either "completed" or "failed"
**Validates: Requirements 4.2, 4.3, 4.4**

Property 6: Date format consistency
*For any* date field in export, the value should be formatted as dd/mm/yyyy
**Validates: Requirements 5.1**

Property 7: Relation name export
*For any* relation field (madrasah, kelas, kamar) in export, the exported value should be the relation name not the ID
**Validates: Requirements 5.2**

Property 8: Gender display transformation
*For any* jenis_kelamin field in export, "L" should be exported as "Laki-laki" and "P" as "Perempuan"
**Validates: Requirements 5.3**

Property 9: Null value handling
*For any* null value in export, it should be exported as empty string not the text "null"
**Validates: Requirements 5.4**

Property 10: Filename pattern compliance
*For any* export file created, the filename should match the pattern "siswa-export-{timestamp}.xlsx"
**Validates: Requirements 6.1, 6.2**

Property 11: File accessibility
*For any* saved export file, it should be retrievable from storage
**Validates: Requirements 6.3**

Property 12: Large dataset queue processing
*For any* export with more than 1000 records, the export should be processed via queue
**Validates: Requirements 7.1**

Property 13: Export retry mechanism
*For any* failed export, the system should retry up to 3 times before marking as permanently failed
**Validates: Requirements 7.4, 7.5**

### Import Properties

Property 14: Import file validation
*For any* uploaded CSV file, format validation should occur before processing begins
**Validates: Requirements 8.2**

Property 15: Missing column detection
*For any* CSV file with incomplete columns, an error should be displayed listing the missing columns
**Validates: Requirements 8.3**

Property 16: Invalid data detection
*For any* CSV row with invalid data, an error should be displayed with the problematic row details
**Validates: Requirements 8.4**

Property 17: Valid import queuing
*For any* valid CSV file, the import should be processed in background via queue
**Validates: Requirements 8.5**

Property 18: Import completion summary
*For any* completed import, a notification should show the count of successful and failed rows
**Validates: Requirements 8.6**

Property 19: Required field validation
*For any* CSV row, if required fields (nama, nisn, madrasah_id, kelas_id, jenis_kelamin, status_tempat_tinggal) are empty, the row should be skipped and logged as failed
**Validates: Requirements 9.1, 9.2**

Property 20: Date validation
*For any* CSV row with invalid date format, the row should be skipped and logged with error
**Validates: Requirements 9.3**

Property 21: Foreign key validation
*For any* CSV row with non-existent madrasah_id or kelas_id, the row should be skipped and logged with error
**Validates: Requirements 9.4**

Property 22: Gender enum validation
*For any* CSV row where jenis_kelamin is not "L" or "P", the row should be skipped and logged with error
**Validates: Requirements 9.5**

Property 23: Failed row reporting
*For any* import with failures, the system should provide a detailed list of failed rows with error messages
**Validates: Requirements 10.1, 10.2**

Property 24: Error log export
*For any* import with errors, an error log CSV file should be downloadable containing row number, data, and error message
**Validates: Requirements 10.3, 10.4**

Property 25: Template header accuracy
*For any* downloaded import template, the CSV headers should match the expected import column names
**Validates: Requirements 11.2**

Property 26: Template round-trip compatibility
*For any* template file downloaded, filling it with valid data and importing should succeed without format errors
**Validates: Requirements 11.4**

## Error Handling

### Export Errors

1. **Queue Connection Failure**
   - Error: Queue system unavailable
   - Handling: Display error notification, suggest trying again later
   - Retry: Automatic retry up to 3 times with exponential backoff

2. **Memory Limit Exceeded**
   - Error: Export dataset too large for available memory
   - Handling: Process in smaller batches, increase batch size dynamically
   - Prevention: Use chunking for datasets > 1000 rows

3. **Storage Write Failure**
   - Error: Cannot write export file to storage
   - Handling: Log error, notify user, mark export as failed
   - Retry: Automatic retry up to 3 times

4. **Invalid Column Selection**
   - Error: Selected column doesn't exist in model
   - Handling: Filter out invalid columns, proceed with valid ones
   - Prevention: Validate column names against model attributes

### Import Errors

1. **Invalid File Format**
   - Error: Uploaded file is not CSV or has wrong encoding
   - Handling: Display error immediately, don't queue import
   - Message: "File must be CSV format with UTF-8 encoding"

2. **Missing Required Columns**
   - Error: CSV missing required headers
   - Handling: Display error with list of missing columns
   - Message: "Missing required columns: nama, nisn, madrasah_id"

3. **Data Validation Failure**
   - Error: Row data doesn't pass validation rules
   - Handling: Skip row, log to failed_import_rows table
   - Report: Include in import summary and error log

4. **Foreign Key Constraint Violation**
   - Error: Referenced madrasah_id or kelas_id doesn't exist
   - Handling: Skip row, log error with specific message
   - Message: "Madrasah with ID 999 not found"

5. **Duplicate NISN**
   - Error: NISN already exists in database
   - Handling: Update existing record instead of creating new
   - Behavior: Upsert based on NISN as unique identifier

6. **Database Connection Failure**
   - Error: Cannot connect to database during import
   - Handling: Fail entire import, mark as failed
   - Retry: Automatic retry up to 3 times

## Testing Strategy

### Unit Testing

We will write unit tests for:

1. **SiswaExporter Column Definitions**
   - Test that all expected columns are defined
   - Test column labels are correct
   - Test state transformations (date format, gender display, etc.)

2. **SiswaImporter Column Definitions**
   - Test that all required columns are defined with validation rules
   - Test validation rules work correctly
   - Test data transformation logic

3. **Import Validation Logic**
   - Test required field validation
   - Test date format validation
   - Test foreign key existence validation
   - Test enum value validation

4. **Export File Naming**
   - Test filename pattern generation
   - Test timestamp format in filename

### Property-Based Testing

We will use **Pest with Property Testing** (via `pest-plugin-faker` or manual property tests) for PHP to verify universal properties.

Each property-based test should run a minimum of 100 iterations to ensure coverage across random inputs.

Property-based tests will be tagged with comments referencing the design document:

```php
// Feature: siswa-export-feature, Property 1: Export job queuing
test('export action dispatches job to queue', function () {
    // Property test implementation
})->repeat(100);
```

Key property tests to implement:

1. **Property 1: Export job queuing** - For any export action, verify job is queued
2. **Property 4: Filtered export accuracy** - For any filter, verify exported data matches filter
3. **Property 6: Date format consistency** - For any date, verify format is dd/mm/yyyy
4. **Property 7: Relation name export** - For any relation, verify name is exported not ID
5. **Property 19: Required field validation** - For any row with empty required fields, verify it's skipped
6. **Property 21: Foreign key validation** - For any non-existent foreign key, verify row is skipped
7. **Property 26: Template round-trip** - For any template, verify it can be imported successfully

### Integration Testing

We will write integration tests for:

1. **Complete Export Flow**
   - Create siswa records
   - Trigger export action
   - Verify job is queued
   - Process job
   - Verify file is created
   - Verify file content matches database

2. **Complete Import Flow**
   - Create CSV file with valid data
   - Upload file
   - Verify import is queued
   - Process job
   - Verify records are created/updated in database
   - Verify import summary is correct

3. **Import with Errors Flow**
   - Create CSV with mix of valid and invalid rows
   - Upload file
   - Process import
   - Verify valid rows are imported
   - Verify invalid rows are logged
   - Verify error log can be downloaded

4. **Template Download and Import**
   - Download template
   - Fill template with sample data
   - Import template
   - Verify import succeeds

## Implementation Notes

### Filament v3 Export/Import System

Filament v3 provides built-in export/import functionality:

- **Exporter**: Extend `Filament\Actions\Exports\Exporter`
- **Importer**: Extend `Filament\Actions\Imports\Importer`
- **Actions**: Use `ExportAction` and `ImportAction` in resource

### Queue Configuration

Ensure queue is configured in `.env`:
```
QUEUE_CONNECTION=database
```

Run queue worker:
```bash
php artisan queue:work
```

### Storage Configuration

Export files will be stored in `storage/app/public/exports/` and accessible via public disk.

Configure in `config/filesystems.php`:
```php
'public' => [
    'driver' => 'local',
    'root' => storage_path('app/public'),
    'url' => env('APP_URL').'/storage',
    'visibility' => 'public',
],
```

### File Cleanup

Implement scheduled command to delete old export files:

```php
// app/Console/Kernel.php
protected function schedule(Schedule $schedule)
{
    $schedule->command('exports:cleanup')->daily();
}
```

### Performance Considerations

1. **Chunking**: Process exports/imports in chunks of 500 rows
2. **Memory**: Use `chunk()` instead of `get()` to avoid loading all records
3. **Timeout**: Set appropriate timeout for queue jobs (e.g., 300 seconds)
4. **Batch Size**: Adjust batch size based on server resources

### Security Considerations

1. **File Upload**: Validate file type and size before processing
2. **CSV Injection**: Sanitize CSV data to prevent formula injection
3. **Access Control**: Ensure only authorized users can export/import
4. **File Storage**: Store export files in non-public directory initially
5. **Download Links**: Use signed URLs with expiration for downloads

## Dependencies

- **Filament v3**: Core framework for admin panel
- **Laravel Queue**: For background job processing
- **Laravel Storage**: For file storage and retrieval
- **Maatwebsite/Laravel-Excel** (optional): For advanced Excel features if needed
- **League/CSV** (optional): For CSV parsing if Filament's built-in is insufficient

## Migration Plan

1. Create `SiswaExporter` class
2. Create `SiswaImporter` class
3. Add export and import actions to `SiswaResource`
4. Test export functionality
5. Test import functionality
6. Create template download functionality
7. Implement file cleanup command
8. Deploy to production
9. Monitor queue performance
10. Gather user feedback and iterate
