# Design Document

## Overview

Desain ini mengubah KelasSeeder dan KamarSeeder dari menggunakan data random/dummy menjadi menggunakan data yang spesifik dan sesuai dengan struktur organisasi madrasah yang sebenarnya. Seeder akan menggunakan mapping berbasis kode madrasah (MTS001, MA001) untuk menentukan struktur kelas dan nama kamar yang tepat.

## Architecture

Arsitektur tetap menggunakan pola Laravel Seeder standar dengan peningkatan:

1. **Data-driven approach**: Menggunakan array konfigurasi yang mendefinisikan struktur kelas per madrasah
2. **Madrasah-specific logic**: Logic berbeda untuk MTS dan MA berdasarkan kode madrasah
3. **Idempotent seeding**: Seeder dapat dijalankan ulang dengan hasil konsisten

### Component Flow

```
DatabaseSeeder
    ↓
MadrasahSeeder (prerequisite)
    ↓
KelasSeeder → Reads Madrasah → Creates Kelas with specific names
    ↓
KamarSeeder → Reads Madrasah → Creates Kamar with companion names
```

## Components and Interfaces

### 1. KelasSeeder

**Responsibility**: Membuat data kelas dengan nama spesifik berdasarkan madrasah

**Key Methods**:
- `run()`: Main execution method
- Menggunakan array konfigurasi untuk mapping kode madrasah ke struktur kelas

**Configuration Structure**:
```php
$kelasConfig = [
    'MTS001' => [
        ['tingkat' => 'VII', 'nama' => 'VII T PA'],
        ['tingkat' => 'VII', 'nama' => 'VII T PI'],
        // ... dst
    ],
    'MA001' => [
        ['tingkat' => 'X', 'nama' => 'X ITT'],
        // ... dst
    ]
];
```

### 2. KamarSeeder

**Responsibility**: Membuat data kamar dengan nama sahabat Nabi

**Key Methods**:
- `run()`: Main execution method
- Menggunakan array konfigurasi untuk mapping kode madrasah ke nama kamar

**Configuration Structure**:
```php
$kamarConfig = [
    'MTS001' => [
        'Abu Bakar',
        'Umar bin Khattab',
        // ... dst
    ],
    'MA001' => [
        'Salman Al-Farisi',
        // ... dst
    ]
];
```

### 3. Factory Integration

KelasFactory dan KamarFactory tetap digunakan untuk field-field lain (wali_kelas_id, kapasitas, dll) namun field nama akan di-override oleh seeder.

## Data Models

### Kelas Model
- `madrasah_id`: Foreign key ke Madrasah
- `tingkat`: String (VII-XII)
- `nama`: String (nama lengkap kelas)
- `wali_kelas_id`: Foreign key ke User (nullable)
- `wali_kamar_id`: Foreign key ke User (nullable)

### Kamar Model
- `madrasah_id`: Foreign key ke Madrasah
- `nama_kamar`: String (nama kamar)
- `jenis_kelamin`: Enum (L/P)
- `kapasitas`: Integer
- `wali_kamar_id`: Foreign key ke User (nullable)
- `user_id`: Foreign key ke User (creator)



## Correctness Properties

*A property is a characteristic or behavior that should hold true across all valid executions of a system-essentially, a formal statement about what the system should do. Properties serve as the bridge between human-readable specifications and machine-verifiable correctness guarantees.*

### Property 1: Madrasah association consistency

*For any* kelas or kamar record created by the seeder, the madrasah_id field should reference an existing Madrasah record in the database.

**Validates: Requirements 1.5, 3.5, 4.4**

### Property 2: Tingkat-nama consistency

*For any* kelas record created by the seeder, the tingkat field should match the level prefix extracted from the nama field (e.g., if nama is "VII T PA", tingkat should be "VII").

**Validates: Requirements 2.5**

### Property 3: Seeder idempotency for Kelas

*For any* madrasah, running KelasSeeder multiple times should produce the same number of kelas records with identical names and tingkat values each time.

**Validates: Requirements 4.1, 4.5**

### Property 4: Seeder idempotency for Kamar

*For any* madrasah, running KamarSeeder multiple times should produce the same number of kamar records with identical names each time.

**Validates: Requirements 4.2, 4.5**

## Error Handling

### Missing Madrasah Data
- **Condition**: No Madrasah records exist in database
- **Response**: Display error message via command output and return early without creating records
- **Validation**: Check `Madrasah::all()->isEmpty()` before proceeding

### Invalid Madrasah Code
- **Condition**: Madrasah code doesn't match any configured codes (MTS001, MA001)
- **Response**: Skip that madrasah (no error, just no records created)
- **Rationale**: Allows flexibility for other madrasah types in future

### Foreign Key Constraints
- **Condition**: wali_kelas_id, wali_kamar_id, or user_id references non-existent User
- **Response**: Use nullable fields or create placeholder users via factory
- **Validation**: Ensure User records exist before assigning

## Testing Strategy

### Unit Tests

Unit tests will verify specific examples and edge cases:

1. **Test MTS001 creates correct kelas structure**
   - Verify exactly 12 kelas created
   - Verify specific names for each tingkat (VII, VIII, IX)
   - Example-based test

2. **Test MA001 creates correct kelas structure**
   - Verify exactly 12 kelas created
   - Verify specific names for each tingkat (X, XI, XII)
   - Example-based test

3. **Test MTS001 creates correct kamar names**
   - Verify exactly 5 kamar created
   - Verify names match companion list for MTS
   - Example-based test

4. **Test MA001 creates correct kamar names**
   - Verify exactly 5 kamar created
   - Verify names match companion list for MA
   - Example-based test

5. **Test error handling when no Madrasah exists**
   - Verify seeder returns early
   - Verify error message is displayed
   - Example-based test

### Property-Based Tests

Property-based tests will verify universal properties across all inputs:

1. **Property Test: Madrasah association consistency**
   - Generate random madrasah records
   - Run seeder
   - Verify all created kelas and kamar have valid madrasah_id references
   - Validates Property 1

2. **Property Test: Tingkat-nama consistency**
   - Run seeder for any madrasah
   - For each kelas, extract prefix from nama and compare with tingkat
   - Validates Property 2

3. **Property Test: Kelas seeder idempotency**
   - Run KelasSeeder twice
   - Compare record counts and data
   - Verify identical results
   - Validates Property 3

4. **Property Test: Kamar seeder idempotency**
   - Run KamarSeeder twice
   - Compare record counts and data
   - Verify identical results
   - Validates Property 4

### Testing Framework

- **Framework**: Pest (Laravel's default testing framework)
- **Database**: SQLite in-memory database for fast test execution
- **Traits**: RefreshDatabase trait to ensure clean state between tests
- **Factories**: Use existing KelasFactory and KamarFactory for test data setup

### Test Execution Strategy

1. Run unit tests first to verify specific examples work correctly
2. Run property-based tests to verify general correctness across scenarios
3. Use database transactions to isolate tests
4. Seed prerequisite data (Madrasah, User) before running seeder tests
