# Design Document: Export PDF Laporan Presensi Asrama

## Overview

Fitur ini mengimplementasikan export PDF untuk Laporan Presensi Asrama dengan mengadaptasi pola yang sudah berhasil diterapkan pada Laporan Presensi Madrasah. Sistem akan menggenerate PDF multi-halaman yang terstruktur per kamar dan per jenis presensi, dengan optimasi query menggunakan aggregate data untuk menghindari masalah performa.

Perbedaan utama dengan laporan madrasah:
- Menggunakan **Kamar** sebagai unit pengelompokan (bukan Kelas)
- Menggunakan **Wali Kamar** (bukan Wali Kelas)
- Terminologi disesuaikan untuk konteks asrama (Santri, Berpuasa, Berhalangan/Haid, dll)
- Tidak ada kolom "Cabut" dan "Terlambat" karena tidak relevan untuk kegiatan asrama

## Architecture

### Component Diagram

```
┌─────────────────────────────────────────────────────────────┐
│  LaporanPresensiAsramaResource                              │
│  ┌───────────────────────────────────────────────────────┐  │
│  │  headerActions: Export PDF Button                     │  │
│  │  - Collect active filters                             │  │
│  │  - Trigger LaporanAsramaExport                        │  │
│  │  - Generate PDF via DomPDF                            │  │
│  │  - Stream download to browser                         │  │
│  └───────────────────────────────────────────────────────┘  │
└─────────────────────────────────────────────────────────────┘
                            │
                            ▼
┌─────────────────────────────────────────────────────────────┐
│  LaporanAsramaExport (implements FromView)                  │
│  ┌───────────────────────────────────────────────────────┐  │
│  │  __construct(array $filters)                          │  │
│  │  view(): View                                         │  │
│  │  prepareData(): array                                 │  │
│  │    - Apply role-based access control                  │  │
│  │    - Fetch Kamar with eager loading                   │  │
│  │    - Fetch JenisPresensiAsrama                        │  │
│  │    - Fetch Siswa per Kamar                            │  │
│  │    - Execute aggregate query for statistics           │  │
│  │    - Transform to lookup array                        │  │
│  └───────────────────────────────────────────────────────┘  │
└─────────────────────────────────────────────────────────────┘
                            │
                            ▼
┌─────────────────────────────────────────────────────────────┐
│  laporan-asrama-pdf.blade.php                               │
│  ┌───────────────────────────────────────────────────────┐  │
│  │  @foreach($kamars as $kamar)                          │  │
│  │    @foreach($jenisPresensis as $jenis)                │  │
│  │      <div class="page-break">                         │  │
│  │        - Header (Kop Surat)                           │  │
│  │        - Report Details                               │  │
│  │        - Data Table                                   │  │
│  │        - Signature Section                            │  │
│  │      </div>                                           │  │
│  │    @endforeach                                        │  │
│  │  @endforeach                                          │  │
│  └───────────────────────────────────────────────────────┘  │
└─────────────────────────────────────────────────────────────┘
```

### Data Flow

1. **User Action**: Pengguna klik "Export PDF" di header table
2. **Filter Collection**: System mengambil filter aktif dari Livewire component
3. **Export Initialization**: LaporanAsramaExport diinstansiasi dengan filters
4. **Data Preparation**: 
   - Query Kamar dengan role-based filtering
   - Query JenisPresensiAsrama
   - Eager load Siswa per Kamar
   - Execute single aggregate query untuk statistik
5. **View Rendering**: Blade template menerima data dan render HTML
6. **PDF Generation**: DomPDF convert HTML ke PDF
7. **Download**: Browser menerima stream download

## Components and Interfaces

### 1. LaporanAsramaExport Class

**Location**: `app/Exports/LaporanAsramaExport.php`

**Responsibilities**:
- Implement `Maatwebsite\Excel\Concerns\FromView` interface
- Process filters dan apply role-based access control
- Execute optimized database queries
- Prepare data structure untuk view

**Public Methods**:
```php
public function __construct(array $filters)
public function view(): View
```

**Protected Methods**:
```php
protected function prepareData(): array
```

**Return Structure dari prepareData()**:
```php
[
    'kamars' => Collection<Kamar>,        // Kamar dengan relasi siswas dan waliKamar
    'jenisPresensis' => Collection<JenisPresensiAsrama>,
    'attendanceData' => [                 // Lookup array untuk performa
        siswa_id => [
            jenis_id => [
                'hadir' => count,
                'sakit' => count,
                'izin' => count,
                'alpha' => count,
            ]
        ]
    ],
    'filters' => array,                   // Original filters
    'user' => User,                       // Current authenticated user
]
```

### 2. Blade Template

**Location**: `resources/views/exports/laporan-asrama-pdf.blade.php`

**Responsibilities**:
- Render HTML structure untuk PDF
- Loop through kamars dan jenisPresensis
- Display data table dengan statistik
- Apply CSS styling untuk print

**Key Sections**:
- Header (Kop Surat)
- Report Details (Laporan, Tanggal, Kamar, Wali Kamar)
- Data Table (NISN, Nama, Statistik, Persentase)
- Signature Section (Kepala Madrasah, Waka Kesiswaan)

### 3. Resource Action

**Location**: `app/Filament/Resources/LaporanPresensiAsramaResource.php`

**Existing Implementation**: Sudah ada di headerActions, perlu dipastikan memanggil class yang benar

**Action Flow**:
```php
Tables\Actions\Action::make('pdf')
    ->action(function ($livewire) {
        // Parse filters
        // Initialize LaporanAsramaExport
        // Generate PDF
        // Stream download
    })
```

## Data Models

### Primary Models

**Kamar**:
- `id`: Primary key
- `madrasah_id`: Foreign key to madrasahs
- `nama_kamar`: Nama kamar
- `jenis_kelamin`: L/P
- `kapasitas`: Integer
- `wali_kamar_id`: Foreign key to users

**Relationships**:
- `belongsTo(Madrasah)`
- `belongsTo(User, 'wali_kamar_id')` as waliKamar
- `hasMany(Siswa)`

**PresensiAsrama**:
- `id`: Primary key
- `tanggal`: Date
- `tahun_ajaran_id`: Foreign key
- `jenis_presensi_asrama_id`: Foreign key
- `kamar_id`: Foreign key
- `is_locked`: Boolean

**Relationships**:
- `belongsTo(TahunAjaran)`
- `belongsTo(JenisPresensiAsrama)`
- `belongsTo(Kamar)`
- `hasMany(DetailPresensiAsrama)`

**DetailPresensiAsrama**:
- `id`: Primary key
- `presensi_asrama_id`: Foreign key
- `siswa_id`: Foreign key
- `status`: enum('hadir', 'sakit', 'izin', 'alpha')
- `keterangan`: Text

**Relationships**:
- `belongsTo(PresensiAsrama)`
- `belongsTo(Siswa)`

**JenisPresensiAsrama**:
- `id`: Primary key
- `nama`: String (contoh: "Puasa Sunnah", "Sholat Berjamaah")
- `kategori`: String
- `sifat`: String
- `madrasah_id`: Foreign key
- `is_active`: Boolean

**Relationships**:
- `belongsTo(Madrasah)`
- `hasMany(PresensiAsrama)`

### Query Optimization Strategy

**Problem**: Loading semua DetailPresensiAsrama models dapat menghasilkan 100k+ records yang membebani memory.

**Solution**: Aggregate query dengan GROUP BY

```sql
SELECT 
    siswa_id, 
    presensi_asramas.jenis_presensi_asrama_id as jenis_id, 
    status, 
    COUNT(*) as total
FROM detail_presensi_asramas
JOIN presensi_asramas ON detail_presensi_asramas.presensi_asrama_id = presensi_asramas.id
WHERE siswa_id IN (...)
  AND [filter conditions]
GROUP BY siswa_id, jenis_id, status
```

**Result**: Hanya ratusan rows aggregate data, bukan puluhan ribu detail records.



## Correctness Properties

*A property is a characteristic or behavior that should hold true across all valid executions of a system-essentially, a formal statement about what the system should do. Properties serve as the bridge between human-readable specifications and machine-verifiable correctness guarantees.*


### Property Reflection

Setelah menganalisis semua acceptance criteria, berikut adalah identifikasi redundansi:

**Redundant Properties yang Dapat Digabung:**

1. **Ordering Properties (2.2, 2.3)**: Kedua property ini bisa digabung menjadi satu property tentang "Data ordering consistency" yang memastikan kamar dan siswa terurut dengan benar.

2. **Label Text Properties (8.1, 8.2, 8.3, 8.5)**: Semua property ini testing label text yang spesifik untuk asrama. Bisa digabung menjadi satu property "Asrama-specific terminology" yang memastikan semua label menggunakan terminologi yang tepat.

3. **CSS Styling Properties (7.2, 7.3, 7.4, 7.5)**: Semua property ini testing CSS rules. Bisa digabung menjadi satu property "PDF styling compliance" yang memastikan semua CSS rules yang required ada.

4. **Role-based Access Properties (6.1, 6.2, 6.3)**: Ketiga property ini testing access control untuk berbagai role. Bisa digabung menjadi satu comprehensive property "Role-based data filtering" yang cover semua role scenarios.

5. **Query Optimization Properties (5.1, 5.2, 5.4, 5.5)**: Property ini semua tentang query optimization. Bisa digabung menjadi satu property "Query optimization compliance" yang memastikan semua best practices diikuti.

**Properties yang Tetap Terpisah:**

- Filter handling (1.3, 6.4): Berbeda scope, satu tentang filter passing, satu tentang filter application
- Percentage calculation (3.3, 3.4): Berbeda concern, satu tentang formula, satu tentang formatting
- Data structure (5.3): Unique property tentang struktur data aggregate
- Page structure (2.1, 2.4, 2.5): Berbeda level, satu tentang pagination, lainnya tentang content per page

**Final Property Count**: Dari 40 acceptance criteria, setelah reflection akan menghasilkan sekitar 15-18 unique properties yang tidak redundant.

Property 1: PDF download response
*For any* export request, the system should return a streamDownload response with PDF content-type and filename matching pattern "Laporan-Presensi-Asrama-YYYY-MM-DD.pdf"
**Validates: Requirements 1.2, 1.4**

Property 2: Filter propagation
*For any* set of active table filters, those filters should be correctly passed to and used by the Export class
**Validates: Requirements 1.3**

Property 3: Role-based data filtering
*For any* authenticated user, the exported data should respect role-based access control: wali kamar sees only their kamar, non-privileged users see only their madrasah, privileged users see all data
**Validates: Requirements 6.1, 6.2, 6.3**

Property 4: Specific kamar filtering
*For any* kamar_id filter value, when specified, only that kamar should be included in the export
**Validates: Requirements 6.4**

Property 5: All kamars when no filter
*For any* user with access to multiple kamars, when no kamar filter is specified, all accessible kamars should be included in the export
**Validates: Requirements 1.5**

Property 6: Page structure per kamar-jenis combination
*For any* set of kamars and jenis presensis, the PDF should contain exactly (kamar_count × jenis_count) pages
**Validates: Requirements 2.1**

Property 7: Data ordering consistency
*For any* set of kamars and siswas, kamars should be ordered by nama_kamar and siswas within each kamar should be ordered by nama
**Validates: Requirements 2.2, 2.3**

Property 8: Complete page header
*For any* page in the PDF, the header should contain judul laporan, nama madrasah, and tahun ajaran
**Validates: Requirements 2.4**

Property 9: Complete report details
*For any* page in the PDF, the details section should contain jenis presensi, kamar, rentang tanggal, and wali kamar
**Validates: Requirements 2.5**

Property 10: Percentage calculation accuracy
*For any* siswa with attendance data, the percentage should equal (total_hadir / (total_hadir + total_sakit + total_izin + total_alpha)) × 100
**Validates: Requirements 3.3**

Property 11: Percentage formatting precision
*For any* calculated percentage, the displayed value should be rounded to exactly 1 decimal place
**Validates: Requirements 3.4**

Property 12: Aggregate data structure
*For any* prepared export data, the attendanceData should be structured as [siswa_id][jenis_id][status] = count
**Validates: Requirements 5.3**

Property 13: Query optimization compliance
*For any* export execution, the system should use a single aggregate query with GROUP BY, eager load waliKamar relation, and avoid loading all detail presensi models
**Validates: Requirements 5.1, 5.2, 5.4, 5.5**

Property 14: Signature data source
*For any* madrasah, the signature section should display nama_kepala, nip_kepala, and nama_waka_kesiswaan from the madrasahs table
**Validates: Requirements 4.2**

Property 15: Date format in signature
*For any* export date, the signature should display the date in "d F Y" format (e.g., "19 December 2025")
**Validates: Requirements 4.4**

Property 16: Asrama-specific terminology
*For any* rendered PDF, labels should use asrama-specific terms: "Nama Santri" (not "Nama Siswa"), "Berpuasa / Hadir" (not just "Hadir"), "Berhalangan (Haid)/Izin" (not just "Izin"), "Tidak Berpuasa/Alpha" (not just "Alpha"), and "REKAPITULASI DAFTAR HADIR SANTRI" as title
**Validates: Requirements 8.1, 8.2, 8.3, 8.4, 8.5**

## Error Handling

### Export Failures

**Scenario**: Export class gagal memproses data
- **Handling**: Catch exception dan return error response dengan message yang jelas
- **User Experience**: Modal error dengan opsi retry

**Scenario**: PDF generation gagal (DomPDF error)
- **Handling**: Log error detail, return user-friendly error message
- **User Experience**: Notifikasi error dengan saran contact admin

### Data Integrity Issues

**Scenario**: Kamar tidak memiliki wali kamar
- **Handling**: Display "-" atau placeholder di kolom wali kamar
- **Impact**: Export tetap berjalan, tidak blocking

**Scenario**: Madrasah tidak memiliki data pejabat (nama_kepala, nip_kepala, nama_waka_kesiswaan)
- **Handling**: Display placeholder "........................."
- **Impact**: Export tetap berjalan dengan placeholder

**Scenario**: Siswa tidak memiliki data presensi sama sekali
- **Handling**: Display 0 untuk semua kolom statistik, 0% untuk persentase
- **Impact**: Siswa tetap muncul di laporan dengan nilai 0

### Access Control Violations

**Scenario**: User tidak memiliki akses ke madrasah manapun
- **Handling**: Return empty dataset, generate PDF dengan message "Tidak ada data"
- **User Experience**: PDF kosong dengan informasi tidak ada data

**Scenario**: Wali kamar tidak memiliki kamar assigned
- **Handling**: Return empty dataset
- **User Experience**: PDF kosong atau notifikasi tidak ada data

### Performance Issues

**Scenario**: Export data sangat besar (ratusan kamar, ribuan siswa)
- **Handling**: Aggregate query sudah optimal, tapi tetap ada timeout risk
- **Mitigation**: Set reasonable timeout, consider background job untuk export besar
- **User Experience**: Loading indicator, timeout message jika terlalu lama

**Scenario**: Memory limit exceeded saat generate PDF
- **Handling**: DomPDF configuration dengan memory limit yang cukup
- **Mitigation**: Optimize image/asset size, consider chunking untuk export sangat besar

## Testing Strategy

### Unit Testing

**LaporanAsramaExport Class**:
- Test `prepareData()` method dengan berbagai filter combinations
- Test role-based filtering logic untuk setiap role
- Test aggregate query structure dan result transformation
- Test edge cases: empty data, missing relations, null values

**Blade Template**:
- Test rendering dengan sample data
- Test loop logic untuk multiple kamars dan jenis presensis
- Test conditional rendering (placeholder vs actual data)
- Test CSS class applications

### Property-Based Testing

Framework: **Pest PHP** dengan **Pest Property Testing Plugin** atau **PHPUnit with QuickCheck**

**Configuration**:
- Minimum 100 iterations per property test
- Use factories untuk generate random test data
- Tag setiap test dengan property number dari design doc

**Key Properties to Test**:

1. **Filter Propagation Property**: Generate random filter combinations, verify they're correctly passed through
2. **Role-based Filtering Property**: Generate users with different roles, verify correct data scoping
3. **Percentage Calculation Property**: Generate random attendance data, verify formula accuracy
4. **Data Structure Property**: Verify attendanceData array structure for any dataset
5. **Ordering Property**: Generate random kamars and siswas, verify correct ordering

**Test Data Generators**:
- `KamarFactory`: Generate kamars with various attributes
- `SiswaFactory`: Generate siswas with kamar assignments
- `PresensiAsramaFactory`: Generate presensi records
- `DetailPresensiAsramaFactory`: Generate detail records with various status
- `UserFactory`: Generate users with different roles and madrasah assignments

### Integration Testing

**Full Export Flow**:
- Test complete flow dari button click sampai PDF download
- Test dengan real database data (seeded)
- Test dengan berbagai user roles
- Test dengan berbagai filter combinations

**PDF Content Validation**:
- Parse generated PDF dan verify content
- Check page count matches expected
- Verify data accuracy dalam PDF
- Check styling dan layout

### Manual Testing Checklist

- [ ] Export dengan filter tahun ajaran
- [ ] Export dengan filter jenis presensi
- [ ] Export dengan filter kamar specific
- [ ] Export dengan filter rentang tanggal
- [ ] Export tanpa filter (all data)
- [ ] Export sebagai super_admin
- [ ] Export sebagai wali kamar
- [ ] Export sebagai user biasa
- [ ] Verify PDF layout di berbagai PDF readers
- [ ] Verify print quality
- [ ] Test dengan data besar (100+ siswa)
- [ ] Test dengan data kosong
- [ ] Test dengan missing data (no wali kamar, no pejabat)

