# Design Document

## Overview

Fitur ini mengupdate bulk action "Naik Kelas" yang sudah ada di `KelasResource/RelationManagers/SiswasRelationManager` untuk menangani kasus khusus kelulusan siswa. Sistem akan mendeteksi apakah siswa berada di kelas akhir (IX atau XII) dan memberikan opsi yang sesuai:
- **Kelas IX (MTs)**: Pilihan lanjut ke MA atau tamat
- **Kelas XII (MA)**: Otomatis tamat
- **Kelas lainnya**: Naik kelas normal (existing behavior)

Design ini memanfaatkan `RiwayatKelas` sebagai single source of truth untuk status siswa.

## Architecture

### Component Structure

```
app/Filament/Resources/
└── KelasResource/
    └── RelationManagers/
        └── SiswasRelationManager.php (update existing)
```

### Flow Diagram

```mermaid
graph TD
    A[Admin clicks Naik Kelas] --> B{Detect Current Level}
    B -->|VII, VIII, X, XI| C[Normal Flow: Show Next Level]
    B -->|IX| D[Graduation Flow MTs]
    B -->|XII| E[Graduation Flow MA]
    
    C --> F[Select Target Class]
    F --> G[Update: status=lulus, create new aktif]
    G --> H[Success Notification]
    
    D --> I{User Choice}
    I -->|Lanjut ke MA| J[Select Class X]
    I -->|Tamat| K[No Target Class]
    J --> L[Update: status=lulus + keterangan, create new aktif]
    K --> M[Update: status=lulus + keterangan, NO new record]
    L --> H
    M --> H
    
    E --> N[Confirm Graduation MA]
    N --> O[Update: status=lulus + keterangan, NO new record]
    O --> H
```

## Components and Interfaces

### 1. Updated Bulk Action: naik_kelas

**Location:** `app/Filament/Resources/KelasResource/RelationManagers/SiswasRelationManager.php`

**Changes:**
- Add graduation detection logic
- Dynamic form based on current class level
- Conditional field visibility
- Enhanced action logic with graduation handling

#### Method: `form()` - Dynamic Form Generation

**Logic:**
```php
1. Get current class level (VII-XII)
2. Determine next level or graduation scenario
3. Build form based on scenario:
   - Normal: Show target class dropdown
   - IX: Show graduation type + conditional target class
   - XII: Show confirmation message only
```

**Form Fields:**

**For Kelas IX:**
```php
Forms\Components\Radio::make('graduation_type')
    ->label('Pilihan Kelulusan')
    ->options([
        'lanjut' => 'Lanjut ke MA',
        'tamat' => 'Tamat Sekolah (Tidak Melanjutkan)'
    ])
    ->required()
    ->reactive()

Forms\Components\Select::make('new_class_id')
    ->label('Pilih Kelas X (MA)')
    ->options(/* Kelas X di madrasah sama */)
    ->required()
    ->visible(fn ($get) => $get('graduation_type') === 'lanjut')

Forms\Components\Textarea::make('keterangan')
    ->label('Keterangan')
    ->placeholder('Catatan tambahan (opsional)')
```

**For Kelas XII:**
```php
Forms\Components\Placeholder::make('info')
    ->label('Informasi')
    ->content('Siswa akan ditandai sebagai LULUS MA dan tidak akan melanjutkan.')

Forms\Components\Textarea::make('keterangan')
    ->label('Keterangan')
    ->placeholder('Catatan tambahan (opsional)')
```

**For Normal Classes:**
```php
Forms\Components\Select::make('new_class_id')
    ->label('Pilih Kelas Tujuan')
    ->options(/* Next level classes */)
    ->required()
```

#### Method: `action()` - Enhanced Action Logic

**Pseudocode:**
```php
function action(Collection $records, array $data) {
    // 1. Validate Tahun Ajaran
    $activeTahunAjaran = TahunAjaran::where('is_aktif', true)->first();
    if (!$activeTahunAjaran) {
        return error_notification();
    }
    
    // 2. Get current class info
    $currentClass = $this->getOwnerRecord();
    $currentLevel = $currentClass->tingkat;
    
    // 3. Determine scenario
    if ($currentLevel === 'IX') {
        return handleGraduationMTs($records, $data, $activeTahunAjaran);
    } elseif ($currentLevel === 'XII') {
        return handleGraduationMA($records, $data, $activeTahunAjaran);
    } else {
        return handleNormalPromotion($records, $data, $activeTahunAjaran);
    }
}
```

### 2. Helper Methods

#### `handleNormalPromotion()`
```php
/**
 * Handle normal class promotion (VII→VIII, VIII→IX, X→XI, XI→XII)
 */
private function handleNormalPromotion($records, $data, $activeTahunAjaran) {
    $newClassId = $data['new_class_id'];
    
    DB::transaction(function () use ($records, $newClassId, $activeTahunAjaran) {
        // Update old RiwayatKelas: status = 'lulus'
        RiwayatKelas::whereIn('siswa_id', $records->pluck('id'))
            ->where('status', 'aktif')
            ->update([
                'status' => 'lulus',
                'keterangan' => 'Naik kelas'
            ]);
        
        // Update Siswa: kelas_id = new class
        Siswa::whereIn('id', $records->pluck('id'))
            ->update(['kelas_id' => $newClassId]);
        
        // Insert new RiwayatKelas: status = 'aktif'
        $riwayatKelasData = $records->map(function ($siswa) use ($newClassId, $activeTahunAjaran) {
            return [
                'siswa_id' => $siswa->id,
                'kelas_id' => $newClassId,
                'tahun_ajaran_id' => $activeTahunAjaran->id,
                'status' => 'aktif',
                'created_at' => now(),
                'updated_at' => now(),
            ];
        })->toArray();
        
        RiwayatKelas::insert($riwayatKelasData);
    });
    
    return success_notification();
}
```

#### `handleGraduationMTs()`
```php
/**
 * Handle MTs graduation (IX → X or Tamat)
 */
private function handleGraduationMTs($records, $data, $activeTahunAjaran) {
    $graduationType = $data['graduation_type'];
    $keterangan = $data['keterangan'] ?? '';
    
    if ($graduationType === 'lanjut') {
        $newClassId = $data['new_class_id'];
        
        DB::transaction(function () use ($records, $newClassId, $activeTahunAjaran, $keterangan) {
            // Update old RiwayatKelas
            RiwayatKelas::whereIn('siswa_id', $records->pluck('id'))
                ->where('status', 'aktif')
                ->update([
                    'status' => 'lulus',
                    'keterangan' => 'Lulus MTs, melanjutkan ke MA. ' . $keterangan
                ]);
            
            // Update Siswa
            Siswa::whereIn('id', $records->pluck('id'))
                ->update(['kelas_id' => $newClassId]);
            
            // Insert new RiwayatKelas for MA
            $riwayatKelasData = $records->map(function ($siswa) use ($newClassId, $activeTahunAjaran) {
                return [
                    'siswa_id' => $siswa->id,
                    'kelas_id' => $newClassId,
                    'tahun_ajaran_id' => $activeTahunAjaran->id,
                    'status' => 'aktif',
                    'created_at' => now(),
                    'updated_at' => now(),
                ];
            })->toArray();
            
            RiwayatKelas::insert($riwayatKelasData);
        });
        
        return success_notification('lanjut_ma', $records->count());
        
    } else { // tamat
        DB::transaction(function () use ($records, $keterangan) {
            // Update old RiwayatKelas
            RiwayatKelas::whereIn('siswa_id', $records->pluck('id'))
                ->where('status', 'aktif')
                ->update([
                    'status' => 'lulus',
                    'keterangan' => 'Lulus MTs, tidak melanjutkan. ' . $keterangan
                ]);
            
            // Update Siswa: set kelas_id = NULL
            Siswa::whereIn('id', $records->pluck('id'))
                ->update(['kelas_id' => null]);
            
            // NO new RiwayatKelas created
        });
        
        return success_notification('tamat_mts', $records->count());
    }
}
```

#### `handleGraduationMA()`
```php
/**
 * Handle MA graduation (XII → Tamat)
 */
private function handleGraduationMA($records, $data, $activeTahunAjaran) {
    $keterangan = $data['keterangan'] ?? '';
    
    DB::transaction(function () use ($records, $keterangan) {
        // Update old RiwayatKelas
        RiwayatKelas::whereIn('siswa_id', $records->pluck('id'))
            ->where('status', 'aktif')
            ->update([
                'status' => 'lulus',
                'keterangan' => 'Lulus MA. ' . $keterangan
            ]);
        
        // Update Siswa: set kelas_id = NULL
        Siswa::whereIn('id', $records->pluck('id'))
            ->update(['kelas_id' => null]);
        
        // NO new RiwayatKelas created
    });
    
    return success_notification('tamat_ma', $records->count());
}
```

## Data Models

### RiwayatKelas Model (existing)
```php
- id: integer
- siswa_id: integer (foreign: siswas)
- kelas_id: integer (foreign: kelas)
- tahun_ajaran_id: integer (foreign: tahun_ajarans)
- status: enum('aktif', 'lulus', 'pindah', 'do')
- keterangan: text (nullable)
- timestamps

Relationships:
- belongsTo(Siswa::class)
- belongsTo(Kelas::class)
- belongsTo(TahunAjaran::class)
```

### Siswa Model (existing)
```php
- id: integer
- kelas_id: integer (nullable) - NULL indicates graduated/not active
- ... other fields

Relationships:
- belongsTo(Kelas::class)
- hasMany(RiwayatKelas::class)
```

### Kelas Model (existing)
```php
- id: integer
- tingkat: enum('VII', 'VIII', 'IX', 'X', 'XI', 'XII')
- nama: string
- madrasah_id: integer
- ... other fields
```

## Correctness Properties

*A property is a characteristic or behavior that should hold true across all valid executions of a system-essentially, a formal statement about what the system should do. Properties serve as the bridge between human-readable specifications and machine-verifiable correctness guarantees.*

### Property 1: Level detection accuracy
*For any* kelas with tingkat in ['VII', 'VIII', 'IX', 'X', 'XI', 'XII'], the system should correctly identify whether it is a final level (IX or XII) or a normal level
**Validates: Requirements 1.1, 1.2, 1.4**

### Property 2: Form field visibility consistency
*For any* graduation scenario, when graduation_type is 'lanjut', the target class field should be visible and required; when graduation_type is 'tamat', the target class field should be hidden
**Validates: Requirements 2.2, 2.3, 2.5**

### Property 3: RiwayatKelas status transition correctness
*For any* student promotion or graduation, the old RiwayatKelas record should have status updated to 'lulus' before any new record is created
**Validates: Requirements 4.1, 4.2, 4.3**

### Property 4: RiwayatKelas creation conditional
*For any* student, a new RiwayatKelas with status 'aktif' should be created if and only if the student is continuing to a next class (not graduating as 'tamat')
**Validates: Requirements 4.4, 4.5**

### Property 5: Siswa kelas_id update correctness
*For any* student who continues to next class, kelas_id should equal the target class ID; for any student who graduates as 'tamat', kelas_id should be NULL
**Validates: Requirements 5.1, 5.2**

### Property 6: Transaction atomicity
*For any* bulk operation, either all students are processed successfully or none are processed (all-or-nothing)
**Validates: Requirements 5.3, 5.4**

### Property 7: Active student query accuracy
*For any* query for active students, the result should include only students who have at least one RiwayatKelas record with status 'aktif'
**Validates: Requirements 8.1, 8.4**

### Property 8: Alumni query accuracy
*For any* query for alumni, the result should include only students who have RiwayatKelas records with status 'lulus' and no RiwayatKelas records with status 'aktif'
**Validates: Requirements 8.2**

## Error Handling

### Validation Errors

1. **No Active Tahun Ajaran**
   - Condition: No TahunAjaran with is_aktif = true
   - Response: Danger notification
   - Message: "Tidak ada Tahun Ajaran yang aktif. Silakan aktifkan satu terlebih dahulu."
   - Action: Cancel operation

2. **No Target Class Available**
   - Condition: No classes found for next level
   - Response: Warning notification
   - Message: "Tidak ada kelas tersedia untuk tingkat [level]."
   - Action: Return empty options

3. **Invalid Graduation Type**
   - Condition: graduation_type not in ['lanjut', 'tamat']
   - Response: Danger notification
   - Message: "Pilihan kelulusan tidak valid."
   - Action: Cancel operation

4. **Missing Target Class for Lanjut**
   - Condition: graduation_type = 'lanjut' but new_class_id is null
   - Response: Form validation error
   - Message: "Kelas tujuan harus dipilih jika siswa melanjutkan."
   - Action: Prevent form submission

### Success Notifications

**Normal Promotion:**
- Title: "Proses Kenaikan Kelas Berhasil"
- Body: "Berhasil menaikkan {count} siswa ke kelas {target_class_name}."

**Graduation MTs - Lanjut:**
- Title: "Proses Kelulusan MTs Berhasil"
- Body: "Berhasil meluluskan {count} siswa dari MTs dan memindahkan ke MA kelas {target_class_name}."

**Graduation MTs - Tamat:**
- Title: "Proses Kelulusan MTs Berhasil"
- Body: "Berhasil meluluskan {count} siswa dari MTs. Siswa tidak melanjutkan ke MA."

**Graduation MA:**
- Title: "Proses Kelulusan MA Berhasil"
- Body: "Berhasil meluluskan {count} siswa dari MA."

## Testing Strategy

### Unit Tests

Unit tests will verify specific scenarios:

1. **Test level detection**
   - Given: Kelas with tingkat 'IX'
   - When: Detect graduation scenario
   - Then: Should return 'graduation_mts'

2. **Test form field visibility**
   - Given: graduation_type = 'lanjut'
   - When: Render form
   - Then: Target class field should be visible

3. **Test RiwayatKelas update for normal promotion**
   - Given: Student in class VII
   - When: Promote to class VIII
   - Then: Old RiwayatKelas status = 'lulus', new RiwayatKelas status = 'aktif'

4. **Test graduation MTs with lanjut**
   - Given: Student in class IX, graduation_type = 'lanjut'
   - When: Process graduation
   - Then: Old RiwayatKelas status = 'lulus' with keterangan, new RiwayatKelas created for class X

5. **Test graduation MTs with tamat**
   - Given: Student in class IX, graduation_type = 'tamat'
   - When: Process graduation
   - Then: Old RiwayatKelas status = 'lulus', no new RiwayatKelas, kelas_id = NULL

6. **Test graduation MA**
   - Given: Student in class XII
   - When: Process graduation
   - Then: Old RiwayatKelas status = 'lulus', no new RiwayatKelas, kelas_id = NULL

### Property-Based Tests

Property-based tests will verify universal properties using **Pest with Pest Property Testing plugin**.

Configuration: Each property test should run a minimum of 100 iterations.

Each property-based test MUST be tagged with: `**Feature: kelulusan-siswa, Property {number}: {property_text}**`

1. **Property Test: Level detection**
   - **Feature: kelulusan-siswa, Property 1: Level detection accuracy**
   - Generate: Random kelas with random tingkat
   - Assert: Detection result matches expected scenario

2. **Property Test: Form visibility**
   - **Feature: kelulusan-siswa, Property 2: Form field visibility consistency**
   - Generate: Random graduation_type
   - Assert: Field visibility matches graduation_type

3. **Property Test: Status transition**
   - **Feature: kelulusan-siswa, Property 3: RiwayatKelas status transition correctness**
   - Generate: Random student with active RiwayatKelas
   - Execute: Promotion/graduation
   - Assert: Old status = 'lulus' before new record created

4. **Property Test: Conditional creation**
   - **Feature: kelulusan-siswa, Property 4: RiwayatKelas creation conditional**
   - Generate: Random graduation scenario
   - Execute: Process graduation
   - Assert: New RiwayatKelas exists iff student continues

5. **Property Test: kelas_id update**
   - **Feature: kelulusan-siswa, Property 5: Siswa kelas_id update correctness**
   - Generate: Random student and graduation scenario
   - Execute: Process graduation
   - Assert: kelas_id matches expected value

6. **Property Test: Active student query**
   - **Feature: kelulusan-siswa, Property 7: Active student query accuracy**
   - Generate: Random mix of active and graduated students
   - Execute: Query active students
   - Assert: Result contains only students with active RiwayatKelas

### Integration Tests

Integration tests will verify complete flows:

1. **Test complete normal promotion flow**
   - Create test data: madrasah, tahun ajaran, kelas, students
   - Execute: Bulk promotion via Livewire component
   - Verify: Database state, notifications

2. **Test complete MTs graduation flow**
   - Create test data with class IX students
   - Execute: Graduation with both 'lanjut' and 'tamat' options
   - Verify: Database state, notifications

3. **Test complete MA graduation flow**
   - Create test data with class XII students
   - Execute: Graduation
   - Verify: Database state, notifications

## Implementation Notes

### Filament 3 Specifics

1. **Dynamic Form Fields**
   - Use `->reactive()` on radio button to trigger visibility changes
   - Use `->visible(fn ($get) => ...)` for conditional field display
   - Use `->required(fn ($get) => ...)` for conditional validation

2. **Form State Management**
   - Use `$get()` closure to access other field values
   - Ensure reactive fields are properly configured

3. **Bulk Action Enhancement**
   - Keep existing bulk action structure
   - Add conditional logic in `form()` and `action()` methods
   - Maintain backward compatibility with normal promotion

### Database Considerations

1. **Transaction Safety**
   - All operations must be wrapped in `DB::transaction()`
   - Ensures data consistency across multiple table updates

2. **Query Optimization**
   - Use `whereIn()` for bulk operations
   - Index on `status` column in `riwayat_kelas` table for faster queries

3. **Data Integrity**
   - Ensure keterangan field is properly populated
   - Maintain referential integrity between Siswa and RiwayatKelas

### Backward Compatibility

1. **Existing Functionality**
   - Normal promotion (VII→VIII, VIII→IX, X→XI, XI→XII) must work exactly as before
   - No breaking changes to existing API

2. **Migration Path**
   - No database migration needed
   - Only code changes in SiswasRelationManager

## Future Enhancements

1. **Bulk Graduation Report**
   - Generate PDF report of graduated students
   - Include statistics and student details

2. **Graduation Certificate Generation**
   - Auto-generate graduation certificates
   - Integrate with document template system

3. **Alumni Management**
   - Dedicated alumni resource
   - Track alumni activities and achievements

4. **Graduation Ceremony Management**
   - Schedule and manage graduation events
   - Track attendance and participation
