# Design Document

## Overview

Fitur ini menambahkan Relation Manager untuk model Kamar di Filament 3, memungkinkan admin untuk melihat dan mengelola siswa yang tinggal di kamar tertentu. Fitur utama adalah bulk action untuk memindahkan siswa antar kamar dengan validasi kapasitas dan jenis kelamin.

Design ini mengikuti pattern yang sudah ada di `SiswasRelationManager` pada `KelasResource`, dengan penyesuaian untuk konteks kamar asrama.

## Architecture

### Component Structure

```
app/Filament/Resources/
└── KamarResource/
    ├── Pages/
    │   ├── ListKamars.php (existing)
    │   ├── CreateKamar.php (existing)
    │   └── EditKamar.php (existing)
    └── RelationManagers/
        └── SiswasRelationManager.php (new)
```

### Flow Diagram

```mermaid
graph TD
    A[Admin opens Kamar detail] --> B[Relation Manager displays Siswas]
    B --> C{Admin selects students}
    C --> D[Click 'Pindah Kamar' bulk action]
    D --> E[System shows form with room options]
    E --> F{Validate gender match}
    F -->|Invalid| G[Show error: gender mismatch]
    F -->|Valid| H{Validate capacity}
    H -->|Exceeds| I[Show error: capacity exceeded]
    H -->|Valid| J[Update kamar_id for selected students]
    J --> K[Show success notification]
    K --> L[Refresh table]
```

## Components and Interfaces

### 1. SiswasRelationManager Class

**Location:** `app/Filament/Resources/KamarResource/RelationManagers/SiswasRelationManager.php`

**Extends:** `Filament\Resources\RelationManagers\RelationManager`

**Properties:**
- `protected static string $relationship = 'siswas'` - Defines the relationship name from Kamar model

**Methods:**

#### `form(Form $form): Form`
Defines the form schema for creating/editing siswa records (minimal implementation).

#### `table(Table $table): Table`
Configures the table display with:
- Columns: NISN, Nama, Jenis Kelamin, Kelas
- Header actions: Bulk action "Pindah Kamar"
- Row actions: Edit, Delete (optional)

#### `getTargetRoomOptions(): array`
Helper method to get available target rooms with validation:
- Same madrasah_id
- Same jenis_kelamin
- Returns array of room_id => room_name

#### `validateRoomCapacity(int $targetRoomId, int $studentsToMove): bool`
Validates if target room has enough capacity:
- Counts current students in target room
- Checks if (current + to_move) <= capacity
- Returns boolean

### 2. Bulk Action: Pindah Kamar

**Configuration:**
- Icon: `heroicon-o-arrow-right-circle`
- Label: "Pindah Kamar"
- Requires selection: Yes
- Deselect after completion: Yes

**Form Schema:**
```php
Forms\Components\Select::make('target_room_id')
    ->label('Pilih Kamar Tujuan')
    ->options(fn () => $this->getTargetRoomOptions())
    ->required()
    ->helperText('Hanya menampilkan kamar dengan jenis kelamin yang sama')
```

**Action Logic:**
1. Get selected students collection
2. Get target room ID from form data
3. Get current room record (owner record)
4. Validate gender match
5. Validate capacity
6. If valid: Update kamar_id for all selected students
7. Show notification (success/error)
8. Refresh table

## Data Models

### Kamar Model (existing)
```php
- id: integer
- madrasah_id: integer
- nama_kamar: string
- jenis_kelamin: enum('Laki-laki', 'Perempuan')
- kapasitas: integer
- wali_kamar_id: integer (nullable)
- user_id: integer

Relationships:
- hasMany(Siswa::class)
- belongsTo(Madrasah::class)
```

### Siswa Model (existing)
```php
- id: integer
- madrasah_id: integer
- kelas_id: integer
- kamar_id: integer (nullable)
- nisn: string
- nama: string
- jenis_kelamin: enum('L', 'P')
- ... (other fields)

Relationships:
- belongsTo(Kamar::class)
- belongsTo(Kelas::class)
```

## Correctness Properties

*A property is a characteristic or behavior that should hold true across all valid executions of a system-essentially, a formal statement about what the system should do. Properties serve as the bridge between human-readable specifications and machine-verifiable correctness guarantees.*

### Property 1: Gender consistency in room transfers
*For any* bulk transfer operation, all target rooms displayed should have the same jenis_kelamin as the source room
**Validates: Requirements 5.1, 5.2, 5.3**

### Property 2: Capacity constraint preservation
*For any* room transfer operation, the total number of students in the target room after transfer should not exceed the room's kapasitas
**Validates: Requirements 3.3**

### Property 3: Madrasah scope preservation
*For any* room transfer operation, all target rooms displayed should belong to the same madrasah_id as the source room
**Validates: Requirements 2.4**

### Property 4: Student room assignment update
*For any* successful bulk transfer, all selected students' kamar_id should be updated to the target room ID
**Validates: Requirements 2.5**

### Property 5: Table refresh after transfer
*For any* successful transfer operation, the relation manager table should display updated data reflecting the new room assignments
**Validates: Requirements 4.3**

## Error Handling

### Validation Errors

1. **No Target Rooms Available**
   - Condition: No rooms with matching gender in same madrasah
   - Response: Show warning notification with message
   - Action: Disable bulk action or show empty dropdown

2. **Capacity Exceeded**
   - Condition: (current_students + students_to_move) > capacity
   - Response: Show danger notification with details
   - Message format: "Kapasitas tidak cukup. Kamar tujuan memiliki X siswa, kapasitas Y, Anda mencoba memindahkan Z siswa."
   - Action: Cancel transfer, keep selection

3. **Gender Mismatch** (should not occur if filtering works)
   - Condition: Target room gender != source room gender
   - Response: Show danger notification
   - Action: Cancel transfer

4. **Database Error**
   - Condition: Exception during update operation
   - Response: Show danger notification with generic error message
   - Action: Rollback transaction, keep selection

### Success Notifications

- Title: "Pemindahan Kamar Berhasil"
- Body: "Berhasil memindahkan {count} siswa ke {room_name}"
- Type: Success
- Duration: 5 seconds

## Testing Strategy

### Unit Tests

Unit tests will verify specific examples and edge cases:

1. **Test empty room transfer**
   - Given: Room with 0 students
   - When: Transfer 5 students
   - Then: All 5 students should be in target room

2. **Test capacity boundary**
   - Given: Room with capacity 10, currently has 8 students
   - When: Transfer 2 students
   - Then: Transfer succeeds
   - When: Transfer 3 students
   - Then: Transfer fails with capacity error

3. **Test gender filtering**
   - Given: Source room with jenis_kelamin "Laki-laki"
   - When: Get target room options
   - Then: Only rooms with "Laki-laki" are returned

### Property-Based Tests

Property-based tests will verify universal properties across all inputs using **Pest with Pest Property Testing plugin** (built on top of PHP's property testing capabilities).

Configuration: Each property test should run a minimum of 100 iterations.

Each property-based test MUST be tagged with a comment explicitly referencing the correctness property in this design document using this format: `**Feature: kamar-siswa-relation-manager, Property {number}: {property_text}**`

1. **Property Test: Gender consistency**
   - **Feature: kamar-siswa-relation-manager, Property 1: Gender consistency in room transfers**
   - Generate: Random source room with random gender
   - Generate: Random set of target rooms
   - Assert: All filtered target rooms have same gender as source

2. **Property Test: Capacity preservation**
   - **Feature: kamar-siswa-relation-manager, Property 2: Capacity constraint preservation**
   - Generate: Random room with random capacity and current students
   - Generate: Random number of students to transfer
   - Assert: If transfer succeeds, total <= capacity; if fails, total > capacity

3. **Property Test: Madrasah scope**
   - **Feature: kamar-siswa-relation-manager, Property 3: Madrasah scope preservation**
   - Generate: Random source room with madrasah_id
   - Generate: Random set of rooms with various madrasah_ids
   - Assert: All filtered target rooms have same madrasah_id as source

4. **Property Test: Student assignment update**
   - **Feature: kamar-siswa-relation-manager, Property 4: Student room assignment update**
   - Generate: Random set of students with initial kamar_id
   - Generate: Random target kamar_id
   - Execute: Transfer operation
   - Assert: All students' kamar_id equals target kamar_id

### Integration Tests

Integration tests will verify the complete flow:

1. **Test complete transfer flow**
   - Create test madrasah, rooms, and students
   - Execute bulk transfer via Livewire component
   - Verify database state
   - Verify notification displayed

2. **Test permission-based access**
   - Test as admin: should see all rooms
   - Test as wali kamar: should only see their rooms

## Implementation Notes

### Filament 3 Specifics

1. **Relation Manager Registration**
   - Add to `KamarResource::getRelations()` method
   - Return array with `SiswasRelationManager::class`

2. **Bulk Action Syntax**
   - Use `Tables\Actions\BulkAction::make('pindah_kamar')`
   - Place in `headerActions()` array
   - Use `deselectRecordsAfterCompletion()` method

3. **Form Components**
   - Use `Forms\Components\Select` for room selection
   - Use closure for dynamic options: `->options(fn () => ...)`
   - Use `->helperText()` for user guidance

4. **Notifications**
   - Use `Filament\Notifications\Notification::make()`
   - Chain methods: `->title()`, `->body()`, `->success()`, `->send()`

### Database Considerations

1. **Transaction Safety**
   - Wrap bulk update in `DB::transaction()`
   - Ensures atomicity of multi-student transfers

2. **Query Optimization**
   - Use `whereIn()` for bulk operations
   - Use `pluck()` to get only needed columns
   - Eager load relationships in table query

3. **Soft Deletes**
   - Both Kamar and Siswa use soft deletes
   - Ensure queries exclude soft-deleted records

### Security Considerations

1. **Authorization**
   - Verify user has permission to manage kamar
   - Respect madrasah scope for multi-tenancy
   - Wali kamar should only manage their assigned rooms

2. **Input Validation**
   - Validate target_room_id exists and is accessible
   - Validate selected students belong to source room
   - Validate all students belong to same madrasah

## Future Enhancements

1. **Room History Tracking**
   - Similar to RiwayatKelas, create RiwayatKamar model
   - Track when students move between rooms
   - Useful for reporting and auditing

2. **Capacity Indicator**
   - Show current occupancy in room selection dropdown
   - Format: "Kamar A (8/10)" indicating 8 students out of 10 capacity

3. **Batch Transfer Wizard**
   - Multi-step form for complex transfers
   - Preview changes before committing
   - Handle capacity overflow by suggesting splits

4. **Notification to Students**
   - Send notification to affected students
   - Inform them of room change
   - Include new room details
