# Design Document: Fix User Export Filament

## Overview

This design document outlines the solution to fix the UserExporter class in Filament that is currently failing silently. The main issues are:

1. Missing madrasah_id column that was recently added to the users table
2. Nullable relations (madrasah) not handled properly
3. Insufficient columns exported (missing roles, timestamps, verification status)
4. No proper error handling causing silent failures with null completed_at

The solution will enhance the UserExporter to include all relevant user data, handle nullable relations gracefully, and ensure proper error reporting.

## Architecture

### Current Architecture
```
UserExporter (Filament Exporter)
  ├── Uses User Model
  ├── Exports only 4 columns: id, name, email, avatar_url
  └── No relation handling
```

### Proposed Architecture
```
UserExporter (Enhanced)
  ├── Uses User Model with eager loading
  ├── Exports 10+ columns including:
  │   ├── Basic fields: id, name, email
  │   ├── Madrasah relation: madrasah.nama (with null handling)
  │   ├── Roles: formatted role names
  │   ├── Timestamps: email_verified_at, created_at, updated_at
  │   └── Avatar: avatar_url
  ├── State formatters for readable output
  └── Proper error handling with logging
```

## Components and Interfaces

### 1. UserExporter Class (Enhanced)

**Location**: `app/Filament/Exports/UserExporter.php`

**Responsibilities**:
- Define export columns with proper labels
- Handle nullable madrasah relation
- Format roles as comma-separated string
- Format dates in readable format
- Format email verification status
- Provide clear notification messages

**Key Methods**:
```php
public static function getColumns(): array
// Returns array of ExportColumn definitions

public static function getCompletedNotificationBody(Export $export): string
// Returns localized notification message
```

### 2. Export Columns Configuration

**Columns to Export**:

| Column | Type | Label | Null Handling |
|--------|------|-------|---------------|
| id | Direct | ID | N/A |
| name | Direct | Nama | N/A |
| email | Direct | Email | N/A |
| madrasah.nama | Relation | Madrasah | Display "-" if null |
| roles | Computed | Role | Display "-" if empty |
| email_verified_at | Direct | Email Terverifikasi | "Belum" if null, "Sudah" if set |
| created_at | Direct | Dibuat Pada | Format: DD/MM/YYYY HH:mm |
| updated_at | Direct | Diperbarui Pada | Format: DD/MM/YYYY HH:mm |
| avatar_url | Direct | Avatar | Display "-" if null |

## Data Models

### User Model (Existing)
```php
class User extends Authenticatable
{
    protected $fillable = [
        'name',
        'email', 
        'password',
        'avatar_url',
        'madrasah_id', // Recently added
    ];
    
    // Relations
    public function madrasah(): BelongsTo
    public function roles(): BelongsToMany
}
```

### Madrasah Model (Existing)
```php
class Madrasah extends Model
{
    protected $fillable = [
        'kode',
        'nama',
        'alamat',
        'logo',
        'nama_kepala',
        'email_kepala',
    ];
}
```


## Correctness Properties

*A property is a characteristic or behavior that should hold true across all valid executions of a system—essentially, a formal statement about what the system should do. Properties serve as the bridge between human-readable specifications and machine-verifiable correctness guarantees.*


### Property 1: Null Safety for Madrasah Relation
*For any* user record with null madrasah_id, exporting that user should complete successfully and display "-" or empty string in the madrasah column, not cause an error or job failure.
**Validates: Requirements 1.1, 1.2**

### Property 2: Madrasah Name Display
*For any* user with assigned madrasah, the exported madrasah column should contain the madrasah's name (string), not the numeric ID.
**Validates: Requirements 1.3, 3.3**

### Property 3: Role Formatting Consistency
*For any* user with roles assigned, the exported roles column should contain all role names separated by commas; for users with no roles, it should display "-".
**Validates: Requirements 3.2**

### Property 4: Date Format Consistency
*For any* user record, all date fields (created_at, updated_at) should be formatted consistently in DD/MM/YYYY HH:mm format when exported.
**Validates: Requirements 3.4**

### Property 5: Email Verification Status Readability
*For any* user record, the email_verified_at column should display "Sudah" if the email is verified or "Belum" if not verified, making the status human-readable.
**Validates: Requirements 3.5**

### Property 6: Export Completion Tracking
*For any* successful export job, the exports table record should have a non-null completed_at timestamp and successful_rows count matching the number of users exported.
**Validates: Requirements 1.4, 4.4**

### Property 7: Export Notification Delivery
*For any* completed export (successful or failed), the system should send a notification to the user who initiated the export with appropriate message including row count.
**Validates: Requirements 1.5, 5.2**

### Property 8: Error Logging and Reporting
*For any* export job that encounters errors, the system should log error details to the application log and identify which rows failed with specific error messages.
**Validates: Requirements 2.1, 2.3, 2.5**

### Property 9: Failed Job Handling
*For any* export job that fails completely, the system should send a failure notification to the user and move the job to the failed_jobs table with error details.
**Validates: Requirements 2.2, 4.5**

### Property 10: Queue Job Lifecycle
*For any* export job dispatched, the system should create a job record in the jobs table, process it appropriately, and remove it from the jobs table upon successful completion.
**Validates: Requirements 4.2, 4.3, 4.4**

### Property 11: Multi-State Export Robustness
*For any* export that includes users with various data states (with/without madrasah, with/without roles, verified/unverified), all rows should export successfully without causing job failure.
**Validates: Requirements 5.5**

### Property 12: Export File Format
*For any* completed export, the downloaded file should be in the correct format (xlsx or csv) with proper MIME type and file extension.
**Validates: Requirements 5.3**

## Error Handling

### 1. Null Relation Handling
**Issue**: User with null madrasah_id causes export to fail
**Solution**: 
- Use Filament's built-in null handling for relations
- Relations automatically return empty string when null
- Add explicit default value using `->default('-')`

### 2. Empty Roles Handling
**Issue**: User with no roles might cause issues
**Solution**:
```php
ExportColumn::make('roles')
    ->label('Role')
    ->state(function (User $record): string {
        $roles = $record->roles->pluck('name')->toArray();
        return empty($roles) ? '-' : implode(', ', $roles);
    })
```

### 3. Export Job Failure Logging
**Issue**: Silent failures with no error messages
**Solution**:
- Filament automatically logs export errors
- Ensure APP_DEBUG=true during development
- Check `storage/logs/laravel.log` for errors
- Failed jobs go to `failed_jobs` table

### 4. Queue Configuration
**Issue**: Queue might not be configured properly
**Solution**:
- Verify QUEUE_CONNECTION=database in .env
- Ensure queue worker is running: `php artisan queue:work`
- For development, can disable queue in exporter

## Testing Strategy

### Unit Tests
We will write unit tests to verify:

1. **Null Madrasah Handling Test**
   - Create user with null madrasah_id
   - Export the user
   - Verify export completes without error
   - Verify madrasah column shows "-" or empty

2. **Role Formatting Test**
   - Create user with multiple roles
   - Export the user
   - Verify roles column contains comma-separated role names

3. **Date Formatting Test**
   - Create user with timestamps
   - Export the user
   - Verify dates are formatted as DD/MM/YYYY HH:mm

### Property-Based Tests
We will use **Pest** (Laravel's testing framework) for property-based testing with a minimum of 100 iterations per test:

1. **Property Test: Null Safety**
   - **Feature: fix-user-export-filament, Property 1: Null Safety for Madrasah Relation**
   - Generate random users with various madrasah_id states (null, valid ID)
   - Export all users
   - Assert all exports complete successfully

2. **Property Test: Role Formatting**
   - **Feature: fix-user-export-filament, Property 3: Role Formatting Consistency**
   - Generate random users with 0-5 random roles
   - Export all users
   - Assert role column format is consistent

3. **Property Test: Date Consistency**
   - **Feature: fix-user-export-filament, Property 4: Date Format Consistency**
   - Generate random users with various timestamps
   - Export all users
   - Assert all date columns match DD/MM/YYYY HH:mm pattern

4. **Property Test: Multi-State Robustness**
   - **Feature: fix-user-export-filament, Property 11: Multi-State Export Robustness**
   - Generate random users with various combinations of data states
   - Export all users
   - Assert all rows export successfully

## Implementation Notes

### Column Definition Best Practices

1. **Use Descriptive Labels**
```php
ExportColumn::make('name')
    ->label('Nama')
```

2. **Handle Relations Safely**
```php
ExportColumn::make('madrasah.nama')
    ->label('Madrasah')
    ->default('-')
```

3. **Format Complex Data**
```php
ExportColumn::make('roles')
    ->label('Role')
    ->state(function (User $record): string {
        return $record->roles->pluck('name')->implode(', ') ?: '-';
    })
```

4. **Format Dates**
```php
ExportColumn::make('created_at')
    ->label('Dibuat Pada')
    ->formatStateUsing(fn ($state) => $state?->format('d/m/Y H:i') ?? '-')
```

5. **Format Boolean/Status Fields**
```php
ExportColumn::make('email_verified_at')
    ->label('Email Terverifikasi')
    ->formatStateUsing(fn ($state) => $state ? 'Sudah' : 'Belum')
```

### Notification Localization

Update notification message to Indonesian:
```php
public static function getCompletedNotificationBody(Export $export): string
{
    $body = 'Export user telah selesai dan ' . 
            number_format($export->successful_rows) . ' ' . 
            'baris berhasil diekspor.';

    if ($failedRowsCount = $export->getFailedRowsCount()) {
        $body .= ' ' . number_format($failedRowsCount) . ' ' . 
                 'baris gagal diekspor.';
    }

    return $body;
}
```
