# Design Document - Fix SiswaExporter Undefined Key Error

## Overview

This design addresses the bug in SiswaExporter where "Undefined array key" errors occur during export processing. The root cause is improper handling of null values and potentially missing eager loading of relationships. The fix will ensure all export columns properly handle null values and use Filament's recommended patterns for data transformation.

## Architecture

### Current Problem

The SiswaExporter is experiencing failures with the error:
```
ErrorException: Undefined array key "nisn"
```

This indicates that the exporter is trying to access array keys that don't exist, likely because:
1. The export is using array access instead of object property access
2. Null values are not being handled properly
3. Relationships may not be eager loaded
4. The state() callbacks may not be checking for null values

### Solution Architecture

```
┌─────────────────────────────────────────────────────────────┐
│                     SiswaExporter                            │
├─────────────────────────────────────────────────────────────┤
│                                                               │
│  getColumns()                                                │
│  ├─ Basic columns (nisn, nis, nik, nama, etc.)              │
│  │  └─ Use direct column names (Filament handles access)    │
│  │                                                            │
│  ├─ Transformed columns (jenis_kelamin, tanggal_lahir)      │
│  │  └─ Use state() with null checks                         │
│  │                                                            │
│  └─ Relation columns (madrasah.nama, kelas.nama)            │
│     └─ Use dot notation (Filament handles eager loading)    │
│                                                               │
│  getModifyQueryUsing()                                       │
│  └─ Eager load relationships: madrasah, kelas, kamar        │
│                                                               │
└─────────────────────────────────────────────────────────────┘
```

## Components and Interfaces

### SiswaExporter Modifications

**Location:** `app/Filament/Exports/SiswaExporter.php`

**Key Changes:**

1. **Add Query Modification for Eager Loading**
   ```php
   public function getModifyQueryUsing(): ?Closure
   {
       return fn ($query) => $query->with(['madrasah', 'kelas', 'kamar']);
   }
   ```

2. **Fix Column Definitions**
   - For simple columns (nisn, nis, nik, etc.): Use column name directly without state() callback
   - For transformed columns: Use state() with proper null handling
   - For relation columns: Use dot notation (madrasah.nama)

3. **Improve State Callbacks**
   - Add null checks before transformations
   - Use null coalescing operator (??) for default values
   - Return empty string for null values

### Column Patterns

**Pattern 1: Simple Columns (No Transformation)**
```php
ExportColumn::make('nisn')
    ->label('NISN'),
```
Filament will automatically handle null values by exporting empty strings.

**Pattern 2: Transformed Columns with Null Handling**
```php
ExportColumn::make('jenis_kelamin')
    ->label('Jenis Kelamin')
    ->state(function (Siswa $record): string {
        if (!$record->jenis_kelamin) {
            return '';
        }
        return $record->jenis_kelamin === 'L' ? 'Laki-laki' : 'Perempuan';
    }),
```

**Pattern 3: Date Columns with Null Handling**
```php
ExportColumn::make('tanggal_lahir')
    ->label('Tanggal Lahir')
    ->state(function (Siswa $record): string {
        return $record->tanggal_lahir
            ? $record->tanggal_lahir->format('d/m/Y')
            : '';
    }),
```

**Pattern 4: Relation Columns**
```php
ExportColumn::make('madrasah.nama')
    ->label('Madrasah'),
```
Filament will automatically handle null relations by exporting empty strings.

## Data Models

### Siswa Model (No Changes Required)

The Siswa model already has proper relationships defined:
- `belongsTo(Madrasah::class)`
- `belongsTo(Kelas::class)`
- `belongsTo(Kamar::class)`

The exporter will leverage these relationships through eager loading.

## Correctness Properties

*A property is a characteristic or behavior that should hold true across all valid executions of a system-essentially, a formal statement about what the system should do. Properties serve as the bridge between human-readable specifications and machine-verifiable correctness guarantees.*

Property 1: Null field handling
*For any* siswa record with null fields, exporting should complete without undefined key errors and export empty strings for null values
**Validates: Requirements 1.1, 1.2, 1.3, 1.4, 1.5**

Property 2: Relationship eager loading
*For any* export operation, the madrasah, kelas, and kamar relationships should be eager loaded before processing records
**Validates: Requirements 2.1, 2.2, 2.3, 2.5**

Property 3: Null relation handling
*For any* siswa record with null kamar relationship, exporting should complete successfully and export empty string for kamar field
**Validates: Requirements 2.4**

Property 4: Date null safety
*For any* siswa record with null tanggal_lahir, exporting should complete without error and export empty string
**Validates: Requirements 3.1, 3.3, 3.4**

Property 5: Date format consistency
*For any* siswa record with non-null tanggal_lahir, the exported value should be formatted as dd/mm/yyyy
**Validates: Requirements 3.2**

Property 6: Gender transformation with null handling
*For any* siswa record, if jenis_kelamin is 'L' export "Laki-laki", if 'P' export "Perempuan", if null export empty string
**Validates: Requirements 4.1, 4.2, 4.3, 4.4**

Property 7: Export completion
*For any* set of siswa records (including those with null values), the export should complete successfully without throwing undefined key errors
**Validates: Requirements 1.5**

## Error Handling

### Undefined Key Errors (Current Bug)

**Problem:** Accessing array keys that don't exist
**Solution:** 
- Use object property access via $record parameter
- Let Filament handle simple column access
- Add explicit null checks in state() callbacks

### Null Pointer Errors

**Problem:** Calling methods on null objects (e.g., `$record->tanggal_lahir->format()`)
**Solution:**
- Check for null before calling methods
- Use ternary operator: `$record->tanggal_lahir ? $record->tanggal_lahir->format('d/m/Y') : ''`

### Missing Relationship Errors

**Problem:** N+1 queries or missing relationship data
**Solution:**
- Implement `getModifyQueryUsing()` to eager load relationships
- Use dot notation for relation columns (Filament handles the rest)

### Empty Result Handling

**Problem:** Export with no records
**Solution:**
- Filament handles this automatically
- Export will create file with headers only

## Testing Strategy

### Unit Testing

We will write unit tests for:

1. **Null Value Handling**
   - Create siswa with null nisn, nis, nik
   - Export and verify empty strings are exported
   - Verify no errors are thrown

2. **Date Formatting with Nulls**
   - Create siswa with null tanggal_lahir
   - Export and verify empty string is exported
   - Create siswa with valid date
   - Export and verify dd/mm/yyyy format

3. **Gender Transformation with Nulls**
   - Create siswa with null jenis_kelamin
   - Export and verify empty string is exported
   - Create siswa with 'L' and 'P'
   - Export and verify "Laki-laki" and "Perempuan"

4. **Relationship Handling**
   - Create siswa without kamar
   - Export and verify empty string for kamar field
   - Create siswa with all relationships
   - Export and verify relation names are exported

### Property-Based Testing

We will use **Pest** for PHP property-based testing.

Each property-based test should run a minimum of 100 iterations.

Property-based tests will be tagged with comments:

```php
// Feature: fix-siswa-exporter-undefined-key, Property 1: Null field handling
test('export handles null fields without errors', function () {
    // Generate random siswa with some null fields
    // Export data
    // Verify no errors and empty strings for nulls
})->repeat(100);
```

Key property tests:

1. **Property 1: Null field handling** - Generate siswa with random null fields, verify export succeeds
2. **Property 4: Date null safety** - Generate siswa with null dates, verify export succeeds
3. **Property 6: Gender transformation** - Generate siswa with various gender values including null, verify correct output
4. **Property 7: Export completion** - Generate any siswa records, verify export always completes

### Integration Testing

We will write integration tests for:

1. **Complete Export with Mixed Data**
   - Create siswa records with mix of null and non-null values
   - Trigger export
   - Process export job
   - Verify file is created
   - Verify file content is correct
   - Verify no errors in queue

2. **Export with All Null Fields**
   - Create siswa with maximum null fields
   - Export and verify success

3. **Export with Missing Relationships**
   - Create siswa without kamar
   - Export and verify success

## Implementation Notes

### Filament Exporter Best Practices

1. **Simple Columns**: Don't use state() callback unless transformation is needed
2. **Null Handling**: Always check for null in state() callbacks
3. **Eager Loading**: Use getModifyQueryUsing() to eager load relationships
4. **Relation Columns**: Use dot notation (relation.field) for automatic handling
5. **Return Types**: State callbacks should return string for consistency

### Testing Approach

1. Fix the exporter code first
2. Write unit tests to verify null handling
3. Write property tests to verify behavior across many inputs
4. Write integration test for complete export flow
5. Manually test with real data

### Deployment Considerations

1. This is a bug fix, deploy as soon as tests pass
2. Clear failed jobs from queue after deployment
3. Monitor queue for any remaining errors
4. Test export functionality in production

## Dependencies

- **Filament v3**: Already installed
- **Laravel Queue**: Already configured
- **Pest**: For testing (should already be installed)

## Migration Plan

1. Fix SiswaExporter code
2. Write and run unit tests
3. Write and run property tests
4. Write and run integration tests
5. Clear failed export jobs from queue
6. Deploy to production
7. Monitor for errors
8. Verify exports work correctly
