# Design Document

## Overview

Fitur customisasi panel navigasi dan tema Filament ini bertujuan untuk memberikan identitas visual yang sesuai dengan brand "AbsensiKauman" dan memperbaiki masalah light mode yang terlalu terang. Design ini mengintegrasikan customisasi native Filament dengan Hasnayeen Themes Plugin yang sudah terpasang, memastikan kedua sistem bekerja harmonis tanpa conflict.

Pendekatan design ini menggunakan prinsip:
1. **Separation of Concerns**: Memisahkan konfigurasi logo/brand, color palette, dan theme switching
2. **Progressive Enhancement**: Membangun di atas Hasnayeen Themes Plugin tanpa merusak fungsionalitas existing
3. **Performance First**: Menggunakan caching dan optimisasi asset untuk performa optimal
4. **Developer Friendly**: Konfigurasi terpusat dan mudah di-maintain

## Architecture

### High-Level Architecture

```
┌─────────────────────────────────────────────────────────────┐
│                     Filament Panel                          │
│  ┌───────────────────────────────────────────────────────┐  │
│  │           AdminPanelProvider (Main Config)            │  │
│  │  ┌─────────────────────────────────────────────────┐  │  │
│  │  │  Brand Configuration                            │  │  │
│  │  │  - Logo (Light & Dark)                          │  │  │
│  │  │  - Brand Name                                   │  │  │
│  │  │  - Favicon                                      │  │  │
│  │  └─────────────────────────────────────────────────┘  │  │
│  │  ┌─────────────────────────────────────────────────┐  │  │
│  │  │  Color Configuration                            │  │  │
│  │  │  - Primary Color (Base)                         │  │  │
│  │  │  - Custom Color Palette                         │  │  │
│  │  └─────────────────────────────────────────────────┘  │  │
│  │  ┌─────────────────────────────────────────────────┐  │  │
│  │  │  Theme Mode Configuration                       │  │  │
│  │  │  - Default Theme Mode                           │  │  │
│  │  │  - Theme Switcher                               │  │  │
│  │  └─────────────────────────────────────────────────┘  │  │
│  └───────────────────────────────────────────────────────┘  │
└─────────────────────────────────────────────────────────────┘
                            │
                            ├──────────────────────────────────┐
                            │                                  │
                ┌───────────▼──────────┐          ┌───────────▼──────────┐
                │  Hasnayeen Themes    │          │  Custom CSS/Tailwind │
                │      Plugin          │          │    Configuration     │
                │                      │          │                      │
                │  - Theme Switching   │          │  - Light Mode Fix    │
                │  - Color Variants    │          │  - Sidebar Styling   │
                │  - User Preferences  │          │  - Custom Components │
                │  - SetTheme MW       │          │  - Responsive Design │
                └──────────────────────┘          └──────────────────────┘
```

### Component Interaction Flow

```
User Request
     │
     ▼
┌─────────────────┐
│  Middleware     │
│  Stack          │
│  ┌───────────┐  │
│  │ SetTheme  │──┼──► Load User Theme Preference
│  └───────────┘  │
└─────────────────┘
     │
     ▼
┌─────────────────┐
│  Panel Config   │
│  ┌───────────┐  │
│  │ Logo      │──┼──► Render Brand Logo
│  │ Colors    │──┼──► Apply Color Palette
│  │ Theme     │──┼──► Apply Theme Mode
│  └───────────┘  │
└─────────────────┘
     │
     ▼
┌─────────────────┐
│  View Rendering │
│  ┌───────────┐  │
│  │ Sidebar   │──┼──► Custom Styled Sidebar
│  │ Content   │──┼──► Themed Content Area
│  │ Components│──┼──► Consistent Styling
│  └───────────┘  │
└─────────────────┘
```

## Components and Interfaces

### 1. AdminPanelProvider Enhancement

**Location**: `app/Providers/Filament/AdminPanelProvider.php`

**Responsibilities**:
- Configure brand logo and name
- Set up color palette
- Register Hasnayeen Themes Plugin
- Configure theme mode defaults

**Key Methods**:

```php
public function panel(Panel $panel): Panel
{
    return $panel
        // Brand Configuration
        ->brandName('AbsensiKauman')
        ->brandLogo(asset('images/logo-light.png'))
        ->darkModeBrandLogo(asset('images/logo-dark.png'))
        ->favicon(asset('images/favicon.png'))
        
        // Color Configuration
        ->colors([
            'primary' => Color::Blue,
            'gray' => Color::Slate,
        ])
        
        // Theme Mode
        ->defaultThemeMode(ThemeMode::Light)
        
        // ... other configurations
}
```

### 2. Theme Configuration

**Location**: `config/themes.php`

**Current Structure**:
```php
return [
    'mode' => 'global', // or 'user' for per-user themes
    'icon' => 'heroicon-o-swatch',
    'default' => [
        'theme' => 'default',
        'theme_color' => 'blue',
    ],
];
```

**Enhanced Structure** (to be implemented):
```php
return [
    'mode' => 'user', // Allow per-user theme selection
    'icon' => 'heroicon-o-swatch',
    'default' => [
        'theme' => 'default',
        'theme_color' => 'blue',
    ],
    
    // Available theme colors
    'colors' => [
        'blue' => [
            'primary' => '#3b82f6',
            'label' => 'Blue',
        ],
        'green' => [
            'primary' => '#10b981',
            'label' => 'Green',
        ],
        'purple' => [
            'primary' => '#8b5cf6',
            'label' => 'Purple',
        ],
        'red' => [
            'primary' => '#ef4444',
            'label' => 'Red',
        ],
    ],
];
```

### 3. Custom CSS for Light Mode Fix

**Location**: `resources/css/filament/admin/theme.css` (to be created)

**Purpose**: Override default Filament light mode colors to be less harsh

**Structure**:
```css
@import '/vendor/filament/filament/resources/css/theme.css';

@config 'tailwind.config.js';

/* Light Mode Customization */
:root {
    /* Softer background colors */
    --gray-50: #f8fafc;   /* Main background - softer than pure white */
    --gray-100: #f1f5f9;  /* Card background */
    --gray-200: #e2e8f0;  /* Border color */
    
    /* Sidebar colors */
    --sidebar-bg-light: #ffffff;
    --sidebar-border-light: #e2e8f0;
    
    /* Content area */
    --content-bg-light: #f8fafc;
}

/* Sidebar styling */
.fi-sidebar {
    background-color: var(--sidebar-bg-light);
    border-right: 1px solid var(--sidebar-border-light);
}

/* Content area */
.fi-main {
    background-color: var(--content-bg-light);
}

/* Card styling */
.fi-section {
    background-color: white;
    border: 1px solid var(--gray-200);
}
```

### 4. Tailwind Configuration Enhancement

**Location**: `tailwind.config.js`

**Enhanced Configuration**:
```javascript
import preset from './vendor/filament/filament/tailwind.config.preset'

export default {
    presets: [preset],
    content: [
        './app/Filament/**/*.php',
        './resources/views/filament/**/*.blade.php',
        './vendor/filament/**/*.blade.php',
        './vendor/hasnayeen/themes/resources/**/*.blade.php',
    ],
    theme: {
        extend: {
            colors: {
                // Custom color palette for light mode
                'light-bg': '#f8fafc',
                'light-card': '#ffffff',
                'light-border': '#e2e8f0',
                
                // Sidebar colors
                'sidebar-light': '#ffffff',
                'sidebar-dark': '#1e293b',
            },
        },
    },
}
```

### 5. Logo Asset Management

**Location**: `public/images/`

**Structure**:
```
public/
└── images/
    ├── logo-light.png      # Logo for light mode
    ├── logo-dark.png       # Logo for dark mode
    ├── logo-collapsed.png  # Logo for collapsed sidebar
    └── favicon.png         # Browser favicon
```

**Requirements**:
- Logo dimensions: 180x60px (recommended)
- Collapsed logo: 40x40px (square)
- Format: PNG with transparency or SVG
- Optimized for web (compressed)

## Data Models

### User Theme Preference

**Table**: `users` (existing table, add column)

**New Column**:
```php
Schema::table('users', function (Blueprint $table) {
    $table->string('theme')->nullable()->default('default');
    $table->string('theme_color')->nullable()->default('blue');
});
```

**Model Enhancement**: `app/Models/User.php`

```php
class User extends Authenticatable
{
    protected $fillable = [
        // ... existing fields
        'theme',
        'theme_color',
    ];
    
    protected $casts = [
        // ... existing casts
    ];
    
    /**
     * Get the user's theme preference
     */
    public function getThemeAttribute($value)
    {
        return $value ?? config('themes.default.theme');
    }
    
    /**
     * Get the user's theme color preference
     */
    public function getThemeColorAttribute($value)
    {
        return $value ?? config('themes.default.theme_color');
    }
}
```

## Correctness Properties

*A property is a characteristic or behavior that should hold true across all valid executions of a system-essentially, a formal statement about what the system should do. Properties serve as the bridge between human-readable specifications and machine-verifiable correctness guarantees.*


### Property Reflection

After analyzing all acceptance criteria, I identified several areas where properties can be consolidated:

**Redundancy Analysis**:
1. Properties 2.2 and 2.3 (file access and format support) can be combined into one comprehensive property about asset handling
2. Properties 7.1, 7.2, 7.3, and 7.4 (color propagation) are all testing the same concept - can be unified
3. Properties 13.3, 13.4, and 13.5 (theme mode behavior) test related but distinct behaviors - keep separate
4. Properties 14.3 and 17.1 (theme persistence) test the same round-trip behavior - can be combined
5. Properties 15.1, 15.2, 15.4, and 15.5 (compatibility) test non-interference - can be consolidated

**Final Property Set**: After consolidation, we have 15 unique, non-redundant properties.

### Correctness Properties

**Property 1: Logo asset accessibility**
*For any* configured logo path (light, dark, or collapsed), the system should be able to access and serve the file in supported formats (PNG, SVG, JPG)
**Validates: Requirements 2.2, 2.3**

**Property 2: Text contrast compliance**
*For any* text element displayed in light mode, the contrast ratio between text and background should meet WCAG AA standards (minimum 4.5:1 for normal text, 3:1 for large text)
**Validates: Requirements 3.4**

**Property 3: Theme change propagation**
*For any* theme configuration change, all components in the application should reflect the new theme without requiring cache clearing or page reload
**Validates: Requirements 4.4, 10.5**

**Property 4: Color palette consistency**
*For any* defined color in the palette (primary, secondary, success, warning, danger), all components of that semantic type should use the exact same color value
**Validates: Requirements 7.1, 7.2, 7.3, 7.4**

**Property 5: Menu icon size consistency**
*For any* set of navigation menu items, all icons should have the same dimensions (width and height)
**Validates: Requirements 8.4**

**Property 6: Menu spacing consistency**
*For any* set of navigation menu items at the same level, the spacing between items should be uniform
**Validates: Requirements 8.5**

**Property 7: Global theme mode uniformity**
*For any* two users when theme mode is set to 'global', both users should see the exact same theme and theme color
**Validates: Requirements 13.3**

**Property 8: User theme mode independence**
*For any* two users when theme mode is set to 'user', each user's theme selection should not affect the other user's theme
**Validates: Requirements 13.4**

**Property 9: Theme color application**
*For any* user-selected theme color, the primary color of all interactive elements (buttons, links, badges) should match the selected color
**Validates: Requirements 13.5, 14.4**

**Property 10: Theme preference persistence (Round-trip)**
*For any* user who selects a theme color, logging out and logging back in should restore the same theme color selection
**Validates: Requirements 14.3, 17.1**

**Property 11: Configuration non-interference**
*For any* custom color palette defined in panel provider, the Hasnayeen Themes Plugin color selection should still function independently without being overridden
**Validates: Requirements 15.1**

**Property 12: Brand configuration persistence across theme changes**
*For any* theme or theme color change, the configured logo and brand name should remain unchanged and visible
**Validates: Requirements 15.2**

**Property 13: Light mode customization universality**
*For any* available theme color in the Hasnayeen Themes Plugin, light mode customizations (softer backgrounds, sidebar styling) should apply consistently
**Validates: Requirements 15.4**

**Property 14: Dark mode customization consistency**
*For any* available theme color in the Hasnayeen Themes Plugin, dark mode customizations should maintain consistent styling and contrast
**Validates: Requirements 15.5**

**Property 15: Theme preference caching**
*For any* user's theme preference, after the first load, subsequent requests should retrieve the preference from cache without querying the database
**Validates: Requirements 17.2**

**Property 16: Cache invalidation on update**
*For any* user who updates their theme preference, the cached value should be immediately invalidated and refreshed with the new value
**Validates: Requirements 17.4**

## Error Handling

### Logo Loading Errors

**Scenario**: Logo file not found or inaccessible

**Handling**:
```php
// In AdminPanelProvider
->brandLogo(function () {
    $logoPath = asset('images/logo-light.png');
    
    // Check if file exists
    if (!file_exists(public_path('images/logo-light.png'))) {
        // Fallback to brand name only
        return null;
    }
    
    return $logoPath;
})
```

**User Experience**: Display brand name as fallback when logo is unavailable

### Theme Preference Loading Errors

**Scenario**: Database error when loading user theme preference

**Handling**:
```php
// In SetTheme middleware or theme service
try {
    $theme = auth()->user()->theme;
    $themeColor = auth()->user()->theme_color;
} catch (\Exception $e) {
    // Log error
    \Log::error('Failed to load theme preference', [
        'user_id' => auth()->id(),
        'error' => $e->getMessage()
    ]);
    
    // Fallback to default
    $theme = config('themes.default.theme');
    $themeColor = config('themes.default.theme_color');
}
```

**User Experience**: Gracefully fallback to default theme without breaking the application

### CSS Compilation Errors

**Scenario**: Custom CSS fails to compile or conflicts with Filament CSS

**Handling**:
- Use separate CSS file for customizations (`theme.css`)
- Import Filament CSS first, then apply overrides
- Use CSS specificity carefully to avoid conflicts
- Test in both development and production builds

**Prevention**:
```css
/* Use higher specificity for overrides */
.fi-sidebar.custom-sidebar {
    /* Custom styles */
}

/* Or use !important sparingly for critical overrides */
:root {
    --custom-bg: #f8fafc !important;
}
```

### Plugin Compatibility Errors

**Scenario**: Hasnayeen Themes Plugin conflicts with custom theme configuration

**Handling**:
- Separate concerns: Plugin handles theme switching, custom config handles branding
- Use plugin's color system for dynamic colors
- Use panel provider for static branding (logo, name)
- Document the separation clearly

**Prevention**:
```php
// Panel Provider - Static branding
->brandName('AbsensiKauman')
->brandLogo(asset('images/logo-light.png'))

// Hasnayeen Plugin - Dynamic theming
ThemesPlugin::make()

// Don't override plugin colors in panel provider
// Let plugin handle color switching
```

## Testing Strategy

### Unit Testing

**Focus Areas**:
1. **Configuration Loading**
   - Test that panel provider loads correct logo paths
   - Test that theme config returns expected values
   - Test fallback values when config is missing

2. **Theme Preference Model**
   - Test User model theme accessors
   - Test default values
   - Test theme preference saving

3. **Middleware Behavior**
   - Test SetTheme middleware applies correct theme
   - Test middleware order in stack
   - Test fallback when user has no preference

**Example Unit Test**:
```php
test('user theme accessor returns default when not set', function () {
    $user = User::factory()->create(['theme' => null]);
    
    expect($user->theme)->toBe(config('themes.default.theme'));
});

test('user can save theme preference', function () {
    $user = User::factory()->create();
    
    $user->update([
        'theme' => 'default',
        'theme_color' => 'green'
    ]);
    
    expect($user->fresh()->theme_color)->toBe('green');
});
```

### Property-Based Testing

**Property Testing Library**: We will use **Pest with Faker** for property-based testing in PHP/Laravel.

**Configuration**: Each property-based test should run a minimum of 100 iterations to ensure comprehensive coverage.

**Property Test Examples**:

```php
// Property 1: Logo asset accessibility
test('logo assets are accessible for all supported formats', function () {
    $formats = ['png', 'svg', 'jpg', 'jpeg'];
    
    foreach ($formats as $format) {
        $logoPath = "images/test-logo.{$format}";
        
        // Create test file
        Storage::disk('public')->put($logoPath, 'test content');
        
        // Test accessibility
        $response = $this->get(asset("storage/{$logoPath}"));
        expect($response->status())->toBe(200);
        
        // Cleanup
        Storage::disk('public')->delete($logoPath);
    }
})->repeat(100);

// Property 4: Color palette consistency
test('all buttons use the same primary color', function () {
    $colors = ['blue', 'green', 'purple', 'red'];
    
    foreach ($colors as $color) {
        // Set theme color
        config(['themes.default.theme_color' => $color]);
        
        // Render multiple button components
        $buttons = [
            Filament::renderComponent('button', ['label' => 'Test 1']),
            Filament::renderComponent('button', ['label' => 'Test 2']),
            Filament::renderComponent('button', ['label' => 'Test 3']),
        ];
        
        // Extract color from each button
        $buttonColors = array_map(function ($html) {
            // Parse HTML and extract color class or style
            return extractPrimaryColor($html);
        }, $buttons);
        
        // All buttons should have the same color
        expect(count(array_unique($buttonColors)))->toBe(1);
    }
})->repeat(100);

// Property 10: Theme preference persistence (Round-trip)
test('theme preference persists across login sessions', function () {
    $user = User::factory()->create();
    $themeColors = ['blue', 'green', 'purple', 'red'];
    
    foreach ($themeColors as $color) {
        // Login
        $this->actingAs($user);
        
        // Set theme preference
        $user->update(['theme_color' => $color]);
        
        // Logout
        auth()->logout();
        
        // Login again
        $this->actingAs($user->fresh());
        
        // Theme preference should be restored
        expect(auth()->user()->theme_color)->toBe($color);
    }
})->repeat(100);

// Property 15: Theme preference caching
test('theme preference uses cache after first load', function () {
    $user = User::factory()->create(['theme_color' => 'blue']);
    
    // First load - should query database
    DB::enableQueryLog();
    $this->actingAs($user);
    $firstLoadQueries = count(DB::getQueryLog());
    DB::disableQueryLog();
    
    // Second load - should use cache
    DB::enableQueryLog();
    $theme = auth()->user()->theme_color;
    $secondLoadQueries = count(DB::getQueryLog());
    DB::disableQueryLog();
    
    // Second load should have fewer queries (cached)
    expect($secondLoadQueries)->toBeLessThan($firstLoadQueries);
})->repeat(100);
```

**Property Test Tags**:
Each property-based test must be tagged with a comment explicitly referencing the correctness property:

```php
/**
 * Feature: filament-navigation-theme-customization, Property 1: Logo asset accessibility
 */
test('logo assets are accessible for all supported formats', function () {
    // ... test implementation
});
```

### Integration Testing

**Focus Areas**:
1. **Full Theme Switching Flow**
   - User selects theme color
   - Preference is saved
   - UI updates without reload
   - Preference persists across sessions

2. **Logo Display Across States**
   - Logo displays in normal sidebar
   - Logo displays in collapsed sidebar
   - Logo displays in mobile view
   - Fallback to brand name when logo missing

3. **Plugin Integration**
   - Hasnayeen Themes Plugin works with custom config
   - Theme switching doesn't break custom branding
   - Multiple users can have different themes (user mode)

**Example Integration Test**:
```php
test('complete theme switching flow works end-to-end', function () {
    $user = User::factory()->create();
    
    // Login
    $this->actingAs($user);
    
    // Visit profile page
    $this->get('/admin/my-profile')
        ->assertOk();
    
    // Change theme color
    $this->post('/admin/my-profile', [
        'theme_color' => 'green'
    ])->assertRedirect();
    
    // Verify preference saved
    expect($user->fresh()->theme_color)->toBe('green');
    
    // Verify UI reflects change (check for green color class)
    $this->get('/admin')
        ->assertSee('green') // or check for specific CSS class
        ->assertOk();
    
    // Logout and login again
    auth()->logout();
    $this->actingAs($user->fresh());
    
    // Verify preference persisted
    $this->get('/admin')
        ->assertSee('green')
        ->assertOk();
});
```

### Manual Testing Checklist

**Visual Testing** (cannot be automated):
1. ✅ Light mode background is softer (not pure white)
2. ✅ Sidebar has good contrast with content area
3. ✅ Logo looks proportional and professional
4. ✅ Menu hover effects are smooth
5. ✅ Theme transitions are not jarring
6. ✅ Mobile responsive design works well
7. ✅ All theme colors look good in both light and dark mode

**Browser Testing**:
- Chrome/Edge (latest)
- Firefox (latest)
- Safari (latest)
- Mobile browsers (iOS Safari, Chrome Mobile)

**Accessibility Testing**:
- Screen reader compatibility
- Keyboard navigation
- Color contrast (automated with axe-core)
- Focus indicators

## Implementation Notes

### Phase 1: Logo and Branding
1. Create logo assets and place in `public/images/`
2. Update `AdminPanelProvider` with logo configuration
3. Test logo display in all states (normal, collapsed, mobile)

### Phase 2: Light Mode Fix
1. Create custom CSS file `resources/css/filament/admin/theme.css`
2. Define softer color palette for light mode
3. Override Filament default colors
4. Update `tailwind.config.js` to include custom CSS
5. Build and test in both dev and production

### Phase 3: Hasnayeen Themes Integration
1. Verify plugin is registered in panel provider
2. Configure theme mode (global vs user)
3. Add theme/theme_color columns to users table
4. Update User model with accessors
5. Test theme switching functionality

### Phase 4: Optimization
1. Implement theme preference caching
2. Optimize logo assets (compress, lazy load)
3. Minify CSS for production
4. Test performance with multiple users

### Phase 5: Documentation
1. Document logo requirements and placement
2. Document theme customization process
3. Document Hasnayeen Themes Plugin integration
4. Create developer guide for future customizations

## Performance Considerations

### Asset Optimization
- **Logo Files**: Compress to < 50KB, use WebP format with PNG fallback
- **CSS**: Minify and purge unused styles in production
- **Caching**: Use browser caching for static assets (1 year expiry)

### Database Optimization
- **Theme Preferences**: Cache user theme preferences (Redis/Memcached)
- **Cache Duration**: 1 hour for theme preferences
- **Cache Invalidation**: Clear cache on theme update

### Expected Performance Metrics
- **Logo Load Time**: < 100ms
- **Theme Switch Time**: < 200ms (without page reload)
- **CSS Load Time**: < 300ms (minified and cached)
- **Database Queries**: 0 additional queries after first load (cached)

## Security Considerations

### Logo Upload (if implemented in future)
- Validate file types (whitelist: PNG, SVG, JPG)
- Validate file size (max 2MB)
- Sanitize SVG files to prevent XSS
- Store in public directory with restricted permissions

### Theme Preference Storage
- Validate theme color against allowed values
- Sanitize user input before saving to database
- Use Eloquent mass assignment protection

### CSS Injection Prevention
- Don't allow user-defined CSS
- Use predefined theme colors only
- Sanitize any dynamic CSS values

## Deployment Checklist

### Pre-Deployment
- [ ] Run all unit tests
- [ ] Run all property-based tests
- [ ] Run integration tests
- [ ] Manual visual testing in all browsers
- [ ] Accessibility audit
- [ ] Performance testing
- [ ] Security review

### Deployment Steps
1. Backup current configuration
2. Deploy logo assets to production
3. Deploy code changes
4. Run migrations (add theme columns)
5. Clear application cache
6. Build production assets (`npm run build`)
7. Clear and warm up cache
8. Verify logo displays correctly
9. Verify theme switching works
10. Monitor for errors

### Post-Deployment
- [ ] Verify logo loads on production
- [ ] Test theme switching with real users
- [ ] Monitor performance metrics
- [ ] Check error logs
- [ ] Gather user feedback

### Rollback Plan
If issues occur:
1. Revert code changes
2. Restore previous assets
3. Clear cache
4. Rebuild assets
5. Verify application works with old configuration

## Future Enhancements

### Potential Features
1. **Custom Theme Builder**: Allow admins to create custom themes via UI
2. **Logo Upload**: Allow logo upload through admin panel
3. **Theme Preview**: Preview theme changes before applying
4. **Scheduled Themes**: Auto-switch themes based on time of day
5. **Theme Marketplace**: Share and download community themes
6. **Advanced Customization**: Custom CSS editor for power users
7. **Theme Analytics**: Track which themes are most popular
8. **Accessibility Themes**: High contrast themes for accessibility

### Technical Debt to Address
1. Remove duplicate navigation group "Settings" in panel provider
2. Consolidate theme configuration if multiple sources exist
3. Optimize middleware stack order
4. Implement proper error boundaries for theme loading
5. Add comprehensive logging for theme-related operations
